package org.kurento.room.internal;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.http.Header;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HTTP;
import org.apache.http.util.EntityUtils;
import com.alibaba.fastjson.JSONObject;

/**
 * 封装了HttpClient的工具类，但HttpClient的功能远不止这些
 * POST、GET、上传下载都测试通过
 *
 * @author cqchen 2015年7月25日11:14:44
 */
public class HttpUtil {
	
	private static final int timeout = 10000;
    private static final int maxConnection = 500;

    private static PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager();
	
    static {
    	connManager.setMaxTotal(maxConnection);
    	connManager.setDefaultMaxPerRoute(maxConnection);
    }
    
    private static final CloseableHttpClient httpClient = HttpClients.custom().setConnectionManager(connManager).setDefaultRequestConfig(
            RequestConfig.custom().setRedirectsEnabled(false).setConnectTimeout(timeout).setSocketTimeout(timeout).
            setConnectionRequestTimeout(timeout).build()).setMaxConnTotal(maxConnection).build();
    
    /**
     * POST请求
     *
     * @param url
     * @param params
     * @param charset
     * @return
     */
    public static String post(String url, Map<String, String> params, String charset) {
    	try {
            HttpPost httpPost = new HttpPost(url);
            List<NameValuePair> formParams = buildNameValuePairParams(params);
            UrlEncodedFormEntity uefEntity = new UrlEncodedFormEntity(formParams, charset);
            httpPost.setEntity(uefEntity);
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                return EntityUtils.toString(response.getEntity());
            }
        } catch (Exception e) {
            //Log.error("Http Post处理失败!", e);
        }
        return null;
    }

    /**
     * POST请求
     *
     * @param url
     * @param params
     * @param headerList
     * @param charset
     * @return
     */
    public static String post(String url, List<Header> headerList, Map<String, String> params, String charset) {
        try {
            HttpPost httpPost = new HttpPost(url);
            List<NameValuePair> formParams = buildNameValuePairParams(params);
            UrlEncodedFormEntity uefEntity = new UrlEncodedFormEntity(formParams, charset);
            httpPost.setEntity(uefEntity);
            if (headerList != null) {
                for (Header header : headerList) {
                    httpPost.addHeader(header);
                }
            }
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                return EntityUtils.toString(response.getEntity());
            }
        } catch (Exception e) {
           // Log.error("Http Post处理失败!", e);
        }
        return null;
    }
    /**
     * post 请求课中和课前的服务
     * @param url	请求地址 
     * @param methodName 请求方法
     * @param params	请求参数
     * @return
     */
    public static JSONObject classPost(String url, String methodName, Map<String, String> params){
		List<Header> headerList = new ArrayList<>();
		headerList.add(new BasicHeader("channel", "0000"));
		headerList.add(new BasicHeader("imei", ""));
		headerList.add(new BasicHeader("platform", "4"));
		headerList.add(new BasicHeader("requestTime", System.currentTimeMillis() + ""));
		headerList.add(new BasicHeader("token", ""));
		headerList.add(new BasicHeader("txcode", methodName));
		headerList.add(new BasicHeader("txversion", "15"));
		headerList.add(new BasicHeader("version", "1.0"));
				
		try {
            HttpPost httpPost = new HttpPost(url);
            List<NameValuePair> formParams = buildNameValuePairParams(params);
            UrlEncodedFormEntity uefEntity = new UrlEncodedFormEntity(formParams, StandardCharsets.UTF_8.name());
            httpPost.setEntity(uefEntity);
            if (headerList != null) {
                for (Header header : headerList) {
                    httpPost.addHeader(header);
                }
            }
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
            	//必须加上utf-8, 防止返回时取数据乱码
                String responseString = EntityUtils.toString(response.getEntity(), "utf-8");
                return JSONObject.parseObject(responseString);
            }
        } catch (Exception e) {
           // Log.error("Http Post处理失败!", e);
        }
        return null;
    }
    /**
     * 直接请求的
     *
     * @param url
     * @param headerList
     * @param requestData
     * @param encodeCharset
     * @param decodeCharset
     * @return
     */
    public static String post(String url, List<Header> headerList, String requestData, String encodeCharset, String decodeCharset) {
        try {
            HttpPost httpPost = new HttpPost(url);
            StringEntity stringEntity = new StringEntity(requestData, (encodeCharset == null ? "UTF-8" : encodeCharset));
            if (headerList != null) {
                for (Header header : headerList) {
                    httpPost.addHeader(header);
                }
            }
            httpPost.setEntity(stringEntity);
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                return EntityUtils.toString(response.getEntity(), (decodeCharset == null ? "UTF-8" : decodeCharset));
            }
        } catch (Exception e) {
            //Log.error("Http Post处理失败!", e);
        }
        return null;
    }
    
    /**
     * 直接URL进行post操作
     * @param url
     * @return
     */
    public static String post(String url) {
        try {
            HttpPost httpPost = new HttpPost(url);
            httpPost.setHeader(HTTP.CONTENT_TYPE, "application/json;UTF-8");
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                return EntityUtils.toString(response.getEntity(), "UTF-8");
            }
        } catch (Exception e) {
           // Log.error("Http Post处理失败!", e);
        }
        return null;
    }

    /**
     * 直接将内容放到post里面
     * @param url
     * @param contentStr
     * @param charset
     * @return
     */
    public static String postContent(String url, String contentStr, String charset) {
        try {
            HttpPost httpPost = new HttpPost(url);
            httpPost.setHeader(HTTP.CONTENT_TYPE, "application/json;UTF-8");
            StringEntity stringEntity = new StringEntity(contentStr, charset);
            httpPost.setEntity(stringEntity);
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                return EntityUtils.toString(response.getEntity(), charset);
            }
        } catch (Exception e) {
            //Log.error("Http Post处理失败!", e);
        }
        return null;
    }
    
    /**
     * http 的 byte[] 方式请求
     * @param url
     * @param bodys
     * @return
     */
    public static byte[] postContent(String url, byte[] bodys) {
    	try {
    		HttpPost httpPost = new HttpPost(url);
    		httpPost.setHeader(HTTP.CONTENT_TYPE, "application/octet-stream");
    		ByteArrayEntity entity = new ByteArrayEntity(bodys);
    		httpPost.setEntity(entity);
    		try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                return EntityUtils.toByteArray(response.getEntity());
            }
    	} catch (Exception e) {
            ///Log.error("Http Post处理失败!", e);
        }
    	return null;
    }

    private static List<NameValuePair> buildNameValuePairParams(Map<String, String> params) {
        List<NameValuePair> nameValuePairParams = new ArrayList<NameValuePair>();
        // 组装POST参数
        if (null != params && !params.isEmpty()) {
            Iterator<Map.Entry<String, String>> iter = params.entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry<String, String> entry = iter.next();
                nameValuePairParams.add(new BasicNameValuePair(entry.getKey(), entry.getValue()));
            }
        }
        return nameValuePairParams;
    }

    /**
     * GET处理
     *
     * @param url
     * @return
     */
    public static String get(String url) {
        return get(url, null, null);
    }
    
    /**
     * 获取byte[]数据，例如图片
     * @param url
     * @return
     */
    public static byte[] getByteData(String url) {
    	try {
    		HttpGet httpGet = new HttpGet(url);
    		try (CloseableHttpResponse response = httpClient.execute(httpGet);) {
                return EntityUtils.toByteArray(response.getEntity());
            }
    	} catch (Exception e) {
           // Log.error("Http Get处理失败!", e);
        }
    	
    	return null;
    }
    
    /**
     * GET操作
     * @param url
     * @param params
     * @param charset
     * @return
     */
    public static String get(String url, List<Header> headerList) {
        try {
            HttpGet httpGet = new HttpGet(url);
            if (headerList != null) {
                for (Header header : headerList) {
                	httpGet.addHeader(header);
                }
            }
            try (CloseableHttpResponse response = httpClient.execute(httpGet);) {
                return EntityUtils.toString(response.getEntity());
            }
        } catch (Exception e) {
            //Log.error("Http Get处理失败!", e);
        }
        return null;
    }
    
    /**
     * get操作
     * @param url
     * @param params
     * @param charset
     * @return
     */
    public static String get(String url, Map<String, String> params, String charset) {
        try {
            List<NameValuePair> nameValuePairParams = buildNameValuePairParams(params);
            String newUrl = url;
            if (null != nameValuePairParams && !nameValuePairParams.isEmpty()) {
                String urlParam = URLEncodedUtils.format(nameValuePairParams, charset);
                if (url.indexOf("?") > -1) {
                    newUrl = url + "&" + urlParam;
                } else {
                    newUrl = url + "?" + urlParam;
                }
            }
            HttpGet httpGet = new HttpGet(newUrl);
            try (CloseableHttpResponse response = httpClient.execute(httpGet);) {
                return EntityUtils.toString(response.getEntity());
            }
        } catch (Exception e) {
           // Log.error("Http Get处理失败!", e);
        }
        return null;
    }
}