package org.kurento.room;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.annotation.PreDestroy;
import org.apache.commons.lang3.RandomStringUtils;
import org.joda.time.DateTime;
import org.kurento.client.IceCandidate;
import org.kurento.client.KurentoClient;
import org.kurento.client.MediaElement;
import org.kurento.client.MediaPipeline;
import org.kurento.client.MediaType;
import org.kurento.commons.PropertiesManager;
import org.kurento.room.api.KurentoClientProvider;
import org.kurento.room.api.KurentoClientSessionInfo;
import org.kurento.room.api.MutedMediaType;
import org.kurento.room.api.RoomHandler;
import org.kurento.room.api.pojo.UserParticipant;
import org.kurento.room.endpoint.SdpType;
import org.kurento.room.exception.RoomException;
import org.kurento.room.exception.RoomException.Code;
import org.kurento.room.internal.DefaultKurentoClientSessionInfo;
import org.kurento.room.internal.Participant;
import org.kurento.room.internal.Room;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Strings;

/**
 * 房间管理对象
 * @author mabaojia
 */
public class RoomManager {
	private final Logger log = LoggerFactory.getLogger(RoomManager.class);
	private static final int SCHEDULE_TIME = 30 * 60 * 1000;//房间销毁时间
	private static final int TIMER_INTERVAL = 60 * 1000;
	private static final int TIMER_DELAY = 5 * 1000;
	private static final int EMPTY_ROOM_LIVE_TIME = 30 * 60 * 1000;
	private final String RECORD_DIR = PropertiesManager.getProperty("app.recorddir", "file:///home/hisun/tmp/");
	private RoomHandler roomHandler;
	private KurentoClientProvider kcProvider;
	private final ConcurrentMap<String, Room> rooms = new ConcurrentHashMap<String, Room>();
	private final ConcurrentMap<String, String> partRooms = new ConcurrentHashMap<String, String>();
	private volatile boolean closed = false;
	private Timer timer;

	/**
	 * 构造函数
	 * @param roomHandler
	 * @param kcProvider
	 */
	public RoomManager(RoomHandler roomHandler, KurentoClientProvider kcProvider) {
		super();
		this.roomHandler = roomHandler;
		this.kcProvider = kcProvider;

		EmptyRoomCleaner task = new EmptyRoomCleaner();
		timer = new Timer();
		
		//每分钟跑一次（第一次执行，启动完成5秒后执行）
		timer.schedule(task, TIMER_DELAY, TIMER_INTERVAL);
	}

	/**
	 * 销毁房间
	 * @param roomId	房间id
	 */
	public void destoryRoom(String roomId) {
		this.getRoomByRoomName(roomId).close();
		rooms.remove(roomId);
	}

	/**
	 * 定时清理房间（分钟执行一次）
	 */
	public class EmptyRoomCleaner extends TimerTask{
		//一分钟遍历一次
		@Override
		public void run() {
			List<String> names = new ArrayList<>();
			log.info("定时清理房间：所有房间信息={}", rooms.toString());
			for (Room r : rooms.values()) {
				// 如果房间从没进过人,则r.hadMember()返回的永远都是false,则超过规定的那个时间后,房间就会销毁
				// 只要房间进过人,哪怕,最后走光了,那么永远返回true,则这里接下来的逻辑就不走了,直接走人员都离开后,如果没人进,则三分钟后销毁的逻辑
				log.info("方法:{}:r.hadMember{}","EmptyRoomCleaner.run()", r.hadMember());
				if (r.hadMember()) {
					continue;
				}
				log.info("方法:{}:{}","EmptyRoomCleaner.run()", "没有走continue");
				long createTime = r.getCreateTime();
				long nowTime = new DateTime().getMillis();

				long period = nowTime - createTime;
				if (period >= EMPTY_ROOM_LIVE_TIME) {
					log.info("方法:{}:{}","EmptyRoomCleaner.run()", r.getName());
					names.add(r.getName());
				}
			}

			for (String name : names) {
				log.info("方法:{}:{},{}分钟","EmptyRoomCleaner.run()", "房间没人进时间到了,销毁房间", EMPTY_ROOM_LIVE_TIME/60/1000);
				RoomManager.this.getRoomByRoomName(name).close();
				rooms.remove(name);
			}
		}
	}

	/**
	 * 用户加入房间
	 * 注：要给房间内所有人发送该用户加入房间的事件通知
	 * @param userName	用户id
	 * @param roomName	房间id
	 * @param webParticipant 
	 * @param role
	 * @param kcSessionInfo		session信息
	 * @param participantId		该用户唯一标识
	 * @return
	 * @throws RoomException	加入异常
	 */
	public Set<UserParticipant> joinRoom(String userName, String role, String roomName, boolean webParticipant,
			KurentoClientSessionInfo kcSessionInfo, String participantId) throws RoomException {
		Room room = rooms.get(roomName);
		if (room == null) {
			log.warn("Room '{}' not found");
			throw new RoomException(Code.ROOM_NOT_FOUND_ERROR_CODE, "用户="+ userName +"加入房间="+ roomName +"，房间不存在");
		}
		if (room.isClosed()) {
			log.warn("用户={}加入房间={}，房间已经关闭", userName, roomName);
			throw new RoomException(Code.ROOM_CLOSED_ERROR_CODE, "用户="+ userName +"加入房间="+ roomName +"，房间已经关闭");
		}
		
		Set<UserParticipant> existingParticipants = getParticipants(roomName);
		
		//判断同一个人不能加入不同的房间
		String joinedRoom = partRooms.get(userName);
//		if (!Strings.isNullOrEmpty(joinedRoom) && !joinedRoom.equals(roomName)) {
		if (!Strings.isNullOrEmpty(joinedRoom)) {
			Room partRoom = rooms.get(joinedRoom);
			if (partRoom != null) {
				Participant pt = partRoom.getParticipantByName(userName);
				try {
					leaveRoom(pt.getId());
				} catch (Exception e) {
					log.warn("'{}'离开房间 '{}' 触发异常", userName, partRoom.getName());
				}
			}
		}

		room.join(participantId, userName, role, webParticipant);
		partRooms.put(userName, roomName);
		return existingParticipants;
	}

	/**
	 * 用户离开房间
	 * @param participantId		
	 * @return 
	 * @throws RoomException
	 */
	public Set<UserParticipant> leaveRoom(String participantId) throws RoomException {
		Participant participant = getParticipant(participantId);
//		String userid = participant.getName();
		Room room = participant.getRoom();
		String roomName = room.getName();
		if (room.isClosed()) {
			log.warn("用户={}离开房间={},房间已经关闭", roomName, participant.getName());
			throw new RoomException(Code.ROOM_CLOSED_ERROR_CODE, "用户="+ roomName +"离开房间="+ participant.getName() +",房间已经关闭");
		}
		room.leave(participantId);
		Set<UserParticipant> remainingParticipants = null;
		try {
			remainingParticipants = getParticipants(roomName);
		} catch (RoomException e) {
			log.debug("Possible collision when closing the room '{}' (not found)");
			remainingParticipants = Collections.emptySet();
		}
		
		//当房间所有人都退出时，调定时任务清除房间
		if (remainingParticipants.isEmpty()) {
			log.info("方法:{}:{},{}分钟","leaveRoom", "房间内的用户都离开了,开启销毁房间的定时", SCHEDULE_TIME/60/1000);
			Timer timer = new Timer();
			RoomCleanerTask cleanTask = new RoomCleanerTask(timer, roomName);
			timer.schedule(cleanTask, SCHEDULE_TIME);
		}
		return remainingParticipants;
	}

	/**
	 * 定时解散房间
	 * @param roomName 房间id
	 * @param StartMillisecond 毫秒
	 */
	public void disbandRoomByRoomName(String roomName, int StartMillisecond) {
		Timer timer = new Timer();
		RoomCleanerTask cleanTask = new RoomCleanerTask(timer, roomName);
		timer.schedule(cleanTask, StartMillisecond);
	}

	/**
	 * 推送媒体流
	 * @param participantId
	 * @param isOffer
	 * @param sdp
	 * @param loopbackAlternativeSrc
	 * @param loopbackConnectionType
	 * @param doLoopback
	 * @param mediaElements
	 * @return 
	 * @throws RoomException
	 */
	public String publishMedia(String participantId, boolean isOffer, String sdp, MediaElement loopbackAlternativeSrc,
			MediaType loopbackConnectionType, boolean doLoopback, MediaElement... mediaElements) throws RoomException {
		log.debug("Request [publishMedia]参数：isOffer={},sdp={},loopbackAltSrc={},lpbkConnType={},doLoopback={},mediaElements={},请求者({})",
				isOffer, sdp, loopbackAlternativeSrc == null, loopbackConnectionType, doLoopback, mediaElements, participantId);

		SdpType sdpType = isOffer ? SdpType.OFFER : SdpType.ANSWER;
		Participant participant = getParticipant(participantId);
		String name = participant.getName();
		Room room = participant.getRoom();
		participant.createPublishingEndpoint();

		for (MediaElement elem : mediaElements) {
			participant.getPublisher().apply(elem);
		}

		String sdpResponse = participant.publishToRoom(sdpType, sdp, doLoopback, loopbackAlternativeSrc, loopbackConnectionType);
		if (sdpResponse == null) {
			throw new RoomException(Code.MEDIA_SDP_ERROR_CODE, "Error generating SDP response for publishing user " + name);
		}

		room.newPublisher(participant);
		return sdpResponse;
	}

	/**
	 * 同上
	 */
	public String publishMedia(String participantId, String sdp, boolean doLoopback, MediaElement... mediaElements) throws RoomException {
		return publishMedia(participantId, true, sdp, null, null, doLoopback, mediaElements);
	}

	/**
	 * 同上
	 */
	public String publishMedia(String participantId, boolean isOffer, String sdp, boolean doLoopback, MediaElement... mediaElements) throws RoomException {
		return publishMedia(participantId, isOffer, sdp, null, null, doLoopback, mediaElements);
	}

	/**
	 * 生成PublishOffer
	 * @param participantId
	 * @return 
	 * @throws RoomException
	 */
	public String generatePublishOffer(String participantId) throws RoomException {
		log.debug("Request [GET_PUBLISH_SDP_OFFER] ({})", participantId);
		Participant participant = getParticipant(participantId);
		String name = participant.getName();
		Room room = participant.getRoom();

		participant.createPublishingEndpoint();

		String sdpOffer = participant.preparePublishConnection();
		if (sdpOffer == null) {
			throw new RoomException(Code.MEDIA_SDP_ERROR_CODE, "Error generating SDP offer for publishing user " + name);
		}

		room.newPublisher(participant);
		return sdpOffer;
	}

	/**
	 *
	 * @param participantId
	 * @throws RoomException
	 */
	public void unpublishMedia(String participantId) throws RoomException {
		log.debug("Request [UNPUBLISH_MEDIA] ({})", participantId);
		Participant participant = getParticipant(participantId);
		if (!participant.isStreaming()) {
			throw new RoomException(Code.USER_NOT_STREAMING_ERROR_CODE, "Participant '" + participant.getName() + "' is not streaming media");
		}
		Room room = participant.getRoom();
		participant.unpublishMedia();
		room.cancelPublisher(participant);
	}

	/**
	 * 订阅视频流
	 * 描述：participantId这个sessionId 订阅 remoteName这个用户id的视频流（如果remoteName是自己，就接受自己的视频流）
	 * 建议： Answer to the peer's request by sending it the
	 * SDP answer generated by the the receiving WebRTC endpoint on the server.
	 * @param remoteName	被订阅者的id
	 * @param sdpOffer		订阅者的offer
	 * @param participantId	订阅者的sessionId
	 * @return
	 * 		由服务器上的 WebRTC endpoint 生成的 SDP应答
	 * @throws RoomException
	 */
	public String subscribe(String remoteName, String sdpOffer, String participantId, Integer rateLevel) throws RoomException {
		log.debug("Request [SUBSCRIBE] remoteParticipant={} sdpOffer={} ({})", remoteName, sdpOffer, participantId);
		Participant participant = getParticipant(participantId);
		//获取订阅者的uid
		String name = participant.getName();
		//获取房间对象
		Room room = participant.getRoom();
		//获取被订阅者的sessionId
		Participant senderParticipant = room.getParticipantByName(remoteName);
		//判断被订阅者是不是存在
		if (senderParticipant == null) {
			log.warn("发布订阅者： {}: 请求被订阅者： {} " + "在房间 {} 没有找到", name, remoteName, room.getName());
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE, "在房间:" + room.getName() + "中没有找到用户：" + remoteName);
		}
		//判断被订阅者的视频流是否存在
		if (!senderParticipant.isStreaming()) {
			log.warn("发布订阅者： {}: 请求 {} 的媒体流，在房间： {} 没有找到", name, remoteName, room.getName());
			throw new RoomException(Code.USER_NOT_STREAMING_ERROR_CODE, "用户 '" + remoteName + " 的视频流在房间： '" + room.getName() + "'没有找到");
		}
		
		String sdpAnswer = participant.receiveMediaFrom(senderParticipant, sdpOffer, rateLevel);
		if (sdpAnswer == null) {
			throw new RoomException(Code.MEDIA_SDP_ERROR_CODE, "用户‘"+ name +"’订阅‘"+ remoteName +"’的时候，无法生成sdp应答.");
		}
		return sdpAnswer;
	}

	/**
	 * 订阅合屏视频流
	 */
	public String subscribeBach(String remoteName, String userId, String sdpOffer, String participantId, Integer rateLevel) throws RoomException {
		log.debug("Request [SUBSCRIBE] remoteParticipant={} sdpOffer={} ({})", remoteName, sdpOffer, participantId);
		Participant participant = getParticipant(participantId);
		//获取订阅者的uid
		String name = participant.getName();
		//获取房间对象
		Room room = participant.getRoom();
		//获取被订阅者的sessionId
		Participant senderParticipant = room.getParticipantByName(remoteName);
		Participant senderParticipant1 = room.getParticipantByName(userId);
		
		//判断被订阅者是不是存在
		if (senderParticipant == null) {
			log.warn("发布订阅者： {}: 请求被订阅者： {} " + "在房间 {} 没有找到", name, remoteName, room.getName());
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE, "在房间:" + room.getName() + "中没有找到用户：" + remoteName);
		}
		//判断被订阅者的视频流是否存在
		if (!senderParticipant.isStreaming()) {
			log.warn("发布订阅者： {}: 请求 {} 的媒体流，在房间： {} 没有找到", name, remoteName, room.getName());
			throw new RoomException(Code.USER_NOT_STREAMING_ERROR_CODE, "用户 '" + remoteName + " 的视频流在房间： '" + room.getName() + "'没有找到");
		}
		
		String sdpAnswer = participant.receiveMediaFromBach(senderParticipant, senderParticipant1, sdpOffer, rateLevel);
		if (sdpAnswer == null) {
			throw new RoomException(Code.MEDIA_SDP_ERROR_CODE, "用户‘"+ name +"’订阅‘"+ remoteName +"’的时候，无法生成sdp应答.");
		}
		return sdpAnswer;
	}
	
	/**
	 * 停止订阅
	 * @param remoteName	被订阅者
	 * @param participantId	订阅者participantId
	 * @throws RoomException
	 */
	public void unsubscribe(String remoteName, String participantId) throws RoomException {
		log.debug("Request [UNSUBSCRIBE] remoteParticipant={} ({})", remoteName, participantId);
		Participant participant = getParticipant(participantId);
		String name = participant.getName();
		Room room = participant.getRoom();
		Participant senderParticipant = room.getParticipantByName(remoteName);
		if (senderParticipant == null) {
			log.warn("PARTICIPANT {}: Requesting to unsubscribe from user {} "
					+ "in room {} but user could not be found", name, remoteName, room.getName());
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE,
					"User " + remoteName + " not found in room " + room.getName());
		}
		participant.cancelReceivingMedia(remoteName);
	}

	/**
	 * Request that carries info about an ICE candidate gathered on the client side.
	 * This information is required to implement the trickle ICE mechanism. Should
	 * be triggered or called whenever an icecandidate event is created by a
	 * RTCPeerConnection.
	 * @param endpointName
	 * @param candidate
	 * @param sdpMLineIndex
	 * @param sdpMid
	 * @param participantId
	 * @throws RoomException
	 */
	public void onIceCandidate(String endpointName, String candidate, int sdpMLineIndex, String sdpMid,
			String participantId) throws RoomException {
		log.debug("Request [ICE_CANDIDATE] endpoint={} candidate={} " + "sdpMLineIdx={} sdpMid={} ({})", endpointName,
				candidate, sdpMLineIndex, sdpMid, participantId);
		Participant participant = getParticipant(participantId);
		participant.addIceCandidate(endpointName, new IceCandidate(candidate, sdpMid, sdpMLineIndex));
	}

	/**
	 * Applies a media element (filter, recorder, mixer, etc.) to media that is
	 * currently streaming or that might get streamed sometime in the future. The
	 * element should have been created using the same pipeline as the publisher's.
	 *
	 * @param participantId		identifier of the owner of the stream
	 * @param element		media element to be added
	 * @throws RoomException
	 */
	public void addMediaElement(String participantId, MediaElement element) throws RoomException {
		addMediaElement(participantId, element, null);
	}

	/**
	 * Applies a media element (filter, recorder, mixer, etc.) to media that is
	 * currently streaming or that might get streamed sometime in the future. The
	 * element should have been created using the same pipeline as the publisher's.
	 * The media connection can be of any type, that is audio, video, data or any
	 * (when the parameter is null).
	 *
	 * @param participantId
	 * @param element	media element to be added
	 * @param type	连接类型
	 * @throws RoomException
	 */
	public void addMediaElement(String participantId, MediaElement element, MediaType type) throws RoomException {
		log.debug("Add media element {} (connection type: {}) to participant {}", element.getId(), type, participantId);
		Participant participant = getParticipant(participantId);
		String name = participant.getName();
		if (participant.isClosed()) {
			throw new RoomException(Code.USER_CLOSED_ERROR_CODE, "Participant '" + name + "' has been closed");
		}
		participant.shapePublisherMedia(element, type);
	}

	/**
	 * 从媒体上断开和删除媒体元素 (filter, recorder, etc.)
	 * @param participantId
	 * @param element media element to be removed
	 * @throws RoomException
	 */
	public void removeMediaElement(String participantId, MediaElement element) throws RoomException {
		log.debug("Remove media element {} from participant {}", element.getId(), participantId);
		Participant participant = getParticipant(participantId);
		String name = participant.getName();
		if (participant.isClosed()) {
			throw new RoomException(Code.USER_CLOSED_ERROR_CODE, "Participant '" + name + "' has been closed");
		}
		participant.getPublisher().revert(element);
	}

	/**
	 * Mutes the streamed media of this publisher in a selective manner.
	 * @param muteType		which leg should be disconnected (audio, video or both)
	 * @param participantId
	 * @throws RoomException
	 */
	public void mutePublishedMedia(MutedMediaType muteType, String participantId) throws RoomException {
		log.debug("Request [MUTE_PUBLISHED] muteType={} ({})", muteType, participantId);
		Participant participant = getParticipant(participantId);
		String name = participant.getName();
		if (participant.isClosed()) {
			throw new RoomException(Code.USER_CLOSED_ERROR_CODE, "Participant '" + name + "' has been closed");
		}
		if (!participant.isStreaming()) {
			throw new RoomException(Code.USER_NOT_STREAMING_ERROR_CODE, "Participant '" + name + "' is not streaming media");
		}
		participant.mutePublishedMedia(muteType);
	}

	/**
	 * Reverts the effects of the mute operation.
	 * @param participantId
	 * @throws RoomException
	 */
	public void unmutePublishedMedia(String participantId) throws RoomException {
		log.debug("Request [UNMUTE_PUBLISHED] muteType={} ({})", participantId);
		Participant participant = getParticipant(participantId);
		String name = participant.getName();
		if (participant.isClosed()) {
			throw new RoomException(Code.USER_CLOSED_ERROR_CODE, "Participant '" + name + "' has been closed");
		}
		if (!participant.isStreaming()) {
			throw new RoomException(Code.USER_NOT_STREAMING_ERROR_CODE, "Participant '" + name + "' is not streaming media");
		}
		participant.unmutePublishedMedia();
	}

	/**
	 * Mutes the incoming media stream from the remote publisher in a selective manner.
	 * @param remoteName
	 * @param muteType
	 * @param participantId
	 * @throws RoomException
	 */
	public void muteSubscribedMedia(String remoteName, MutedMediaType muteType, String participantId)
			throws RoomException {
		log.debug("Request [MUTE_SUBSCRIBED] remoteParticipant={} muteType={} ({})", remoteName, muteType, participantId);
		Participant participant = getParticipant(participantId);
		String name = participant.getName();
		Room room = participant.getRoom();
		Participant senderParticipant = room.getParticipantByName(remoteName);
		if (senderParticipant == null) {
			log.warn("PARTICIPANT {}: Requesting to mute streaming from {} " + "in room {} but user could not be found",
					name, remoteName, room.getName());
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE, "User " + remoteName + " not found in room " + room.getName());
		}
		if (!senderParticipant.isStreaming()) {
			log.warn("PARTICIPANT {}: Requesting to mute streaming from {} "
					+ "in room {} but user is not streaming media", name, remoteName, room.getName());
			throw new RoomException(Code.USER_NOT_STREAMING_ERROR_CODE,
					"User '" + remoteName + " not streaming media in room '" + room.getName() + "'");
		}
		participant.muteSubscribedMedia(senderParticipant, muteType);
	}

	/**
	 * Reverts any previous mute operation.
	 * @param remoteName
	 * @param participantId
	 * @throws RoomException
	 */
	public void unmuteSubscribedMedia(String remoteName, String participantId) throws RoomException {
		log.debug("Request [UNMUTE_SUBSCRIBED] remoteParticipant={} ({})", remoteName, participantId);
		Participant participant = getParticipant(participantId);
		String name = participant.getName();
		Room room = participant.getRoom();
		Participant senderParticipant = room.getParticipantByName(remoteName);
		if (senderParticipant == null) {
			log.warn("PARTICIPANT {}: Requesting to unmute streaming from {} "
					+ "in room {} but user could not be found", name, remoteName, room.getName());
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE, "User " + remoteName + " not found in room " + room.getName());
		}
		if (!senderParticipant.isStreaming()) {
			log.warn("PARTICIPANT {}: Requesting to unmute streaming from {} "
					+ "in room {} but user is not streaming media", name, remoteName, room.getName());
			throw new RoomException(Code.USER_NOT_STREAMING_ERROR_CODE,
					"User '" + remoteName + " not streaming media in room '" + room.getName() + "'");
		}
		participant.unmuteSubscribedMedia(senderParticipant);
	}

	// ----------------- ADMIN (DIRECT or SERVER-SIDE) REQUESTS ------------
	/**
	 * Closes all resources. This method has been annotated with the @PreDestroy
	 * directive (javax.annotation package) so that it will be automatically called
	 * when the RoomManager instance is container-managed. <br/>
	 * <strong>Dev advice:</strong> Send notifications to all participants to inform
	 * that their room has been forcibly closed.
	 *
	 * @see RoomManager#closeRoom(String)
	 */
	@PreDestroy
	public void close() {
		closed = true;
		log.info("Closing all rooms");
		for (String roomName : rooms.keySet()) {
			try {
				closeRoom(roomName);
			} catch (Exception e) {
				log.warn("Error closing room '{}'", roomName, e);
			}
		}
	}

	/**
	 * @return true after {@link #close()} has been called
	 */
	public boolean isClosed() {
		return closed;
	}

	/**
	 * Returns all currently active (opened) rooms.
	 * @return set of the rooms' identifiers (names)
	 */
	public Set<String> getRooms() {
		return new HashSet<String>(rooms.keySet());
	}

	/**
	 * Returns all the participants inside a room.
	 * @param roomName
	 * @return 
	 * @throws RoomException
	 */
	public Set<UserParticipant> getParticipants(String roomName) throws RoomException {
		Room room = rooms.get(roomName);
		if (room == null) {
			throw new RoomException(Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
		}
		Collection<Participant> participants = room.getParticipants();
		Set<UserParticipant> userParts = new HashSet<UserParticipant>();
		// 将Participant对象封装成UserParticipant对象
		for (Participant p : participants) {
			if (!p.isClosed()) {
				// userParts.add(new UserParticipant(p.getId(), p.getName(), p.isStreaming()));
				UserParticipant u = new UserParticipant(p.getId(), p.getName(), p.isStreaming());
				u.setRole(p.getRole());
				userParts.add(u);
			}
		}
		return userParts;
	}

	/**
	 * Returns all the publishers (participants streaming their media) inside a room.
	 * @param roomName
	 * @return set of {@link UserParticipant} POJOS representing the existing publishers
	 * @throws RoomException
	 */
	public Set<UserParticipant> getPublishers(String roomName) throws RoomException {
		Room r = rooms.get(roomName);
		if (r == null) {
			throw new RoomException(Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
		}
		Collection<Participant> participants = r.getParticipants();
		Set<UserParticipant> userParts = new HashSet<UserParticipant>();
		for (Participant p : participants) {
			if (!p.isClosed() && p.isStreaming()) {
				userParts.add(new UserParticipant(p.getId(), p.getName(), true));
			}
		}
		return userParts;
	}

	/**
	 * Returns all the subscribers (participants subscribed to a least one stream of
	 * another user) inside a room. A publisher which subscribes to its own stream
	 * (loopback) and will not be included in the returned values unless it requests
	 * explicitly a connection to another user's stream.
	 *
	 * @param roomName
	 * @return set of {@link UserParticipant} POJOS representing the existing subscribers
	 * @throws RoomException
	 */
	public Set<UserParticipant> getSubscribers(String roomName) throws RoomException {
		Room r = rooms.get(roomName);
		if (r == null) {
			throw new RoomException(Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
		}
		Collection<Participant> participants = r.getParticipants();
		Set<UserParticipant> userParts = new HashSet<UserParticipant>();
		for (Participant p : participants) {
			if (!p.isClosed() && p.isSubscribed()) {
				userParts.add(new UserParticipant(p.getId(), p.getName(), p.isStreaming()));
			}
		}
		return userParts;
	}

	/**
	 * Returns the peer's publishers (participants from which the peer is receiving
	 * media). The own stream doesn't count.
	 *
	 * @param participantId
	 * @return set of {@link UserParticipant} POJOS representing the publishers this participant is currently subscribed to
	 * @throws RoomException
	 */
	public Set<UserParticipant> getPeerPublishers(String participantId) throws RoomException {
		Participant participant = getParticipant(participantId);
		if (participant == null) {
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE,
					"No participant with id '" + participantId + "' was found");
		}
		Set<String> subscribedEndpoints = participant.getConnectedSubscribedEndpoints();
		Room room = participant.getRoom();
		Set<UserParticipant> userParts = new HashSet<UserParticipant>();
		for (String epName : subscribedEndpoints) {
			Participant p = room.getParticipantByName(epName);
			userParts.add(new UserParticipant(p.getId(), p.getName()));
		}
		return userParts;
	}

	public void setParticipantRole(String userName, String role, String callerID) {
		Participant caller = getParticipant(callerID);
		Room room = caller.getRoom();
		Participant callee = room.getParticipantByName(userName);
		callee.setRole(role);
	}

	public String getParticipantRole(String pid) {
		String role = null;
		Participant participant = getParticipant(pid);
		if (participant != null) {
			role = participant.getRole();
		}

		return role;
	}

	public String getParticipantIdByName(String userName, String participantId) {
		Participant participant = getParticipant(participantId);
		Room room = participant.getRoom();
		Participant senderParticipant = room.getParticipantByName(userName);
		String id = null;
		if (senderParticipant != null) {
			id = senderParticipant.getId();
		}

		return id;
	}

	public String getTeacherGroupId(String pid) {
		Participant participant = getParticipant(pid);
		Room room = participant.getRoom();
		String teacherGroupId = room.getTeacherInGroupId();
		return teacherGroupId;
	}

	public String getParticipantVideoStatus(String pid) {
		String videoStatus = null;
		Participant participant = getParticipant(pid);
		if (participant != null) {
			videoStatus = participant.getPeopleVideoStatus();
		}
		return videoStatus;
	}

	public void setParticipantVideoStatus(String userName, String videoStatus, String callerID) {
		Participant caller = getParticipant(callerID);
		Room room = caller.getRoom();
		Participant callee = room.getParticipantByName(userName);
		callee.setPeopleVideoStatus(videoStatus);
	}

	/**
	 * Returns the peer's subscribers (participants towards the peer is streaming
	 * media). The own stream doesn't count.
	 * @param participantId
	 * @return set of {@link UserParticipant} POJOS representing the participants
	 *         subscribed to this peer
	 * @throws RoomException
	 */
	public Set<UserParticipant> getPeerSubscribers(String participantId) throws RoomException {
		Participant participant = getParticipant(participantId);
		if (participant == null) {
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE,
					"No participant with id '" + participantId + "' was found");
		}
		if (!participant.isStreaming()) {
			throw new RoomException(Code.USER_NOT_STREAMING_ERROR_CODE,
					"Participant with id '" + participantId + "' is not a publisher yet");
		}
		Set<UserParticipant> userParts = new HashSet<UserParticipant>();
		Room room = participant.getRoom();
		String endpointName = participant.getName();
		for (Participant p : room.getParticipants()) {
			if (p.equals(participant)) {
				continue;
			}
			Set<String> subscribedEndpoints = p.getConnectedSubscribedEndpoints();
			if (subscribedEndpoints.contains(endpointName)) {
				userParts.add(new UserParticipant(p.getId(), p.getName()));
			}
		}
		return userParts;
	}

	/**
	 * Checks if a participant is currently streaming media.
	 * @param participantId
	 * @return true if the participant is streaming media, false otherwise
	 * @throws RoomException
	 */
	public boolean isPublisherStreaming(String participantId) throws RoomException {
		Participant participant = getParticipant(participantId);
		if (participant == null) {
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE,
					"No participant with id '" + participantId + "' was found");
		}
		if (participant.isClosed()) {
			throw new RoomException(Code.USER_CLOSED_ERROR_CODE,
					"Participant '" + participant.getName() + "' has been closed");
		}
		return participant.isStreaming();
	}

	public boolean isRoomExist(String roomid) {
		Room room = rooms.get(roomid);
		if (room != null) {
			return true;
		}

		return false;
	}

	/**
	 * 创建房间
	 * @param kcSessionInfo
	 *            bean that will be passed to the {@link KurentoClientProvider} in
	 *            order to obtain the {@link KurentoClient} that will be used by the room
	 * @throws RoomException
	 */
	public void createRoom(KurentoClientSessionInfo kcSessionInfo) throws RoomException {
		String roomName = kcSessionInfo.getRoomName();
		Room room = rooms.get(kcSessionInfo);
		if (room != null) {
			throw new RoomException(Code.ROOM_CANNOT_BE_CREATED_ERROR_CODE, "Room '" + roomName + "' already exists");
		}
		KurentoClient kurentoClient = kcProvider.getKurentoClient(kcSessionInfo);

		room = new Room(null, roomName, kurentoClient, roomHandler, kcProvider.destroyWhenUnused());

		Room oldRoom = rooms.putIfAbsent(roomName, room);
		if (oldRoom != null) {
			log.warn("Room '{}' has just been created by another thread", roomName);
			return;
		}
		String kcName = "[NAME NOT AVAILABLE]";
		if (kurentoClient.getServerManager() != null) {
			kcName = kurentoClient.getServerManager().getName();
		}
		log.warn("No room '{}' exists yet. Created one " + "using KurentoClient '{}'.", roomName, kcName);
	}

	public Set<String> queryOnGoingRoom() {
		return rooms.keySet();
	}

	/**
	 * 创建房间
	 * @param uid
	 * @param roomId
	 * @throws RoomException
	 */
	public void createRoom(String uid, String roomId) throws RoomException {
		Room room = rooms.get(roomId);
		if (room != null) {
			throw new RoomException(Code.ROOM_CANNOT_BE_CREATED_ERROR_CODE, "Room '" + roomId + "' already exists");
		}
		KurentoClientSessionInfo kcSessionInfo = new DefaultKurentoClientSessionInfo(uid, roomId);
		KurentoClient kurentoClient = kcProvider.getKurentoClient(kcSessionInfo);

		// FIXME: add by wangjinglong here should return the message...
		if (kurentoClient == null) {
			throw new RoomException(Code.KMS_NOT_READY, "applying the media server...");
		}

		room = new Room(uid, roomId, kurentoClient, roomHandler, kcProvider.destroyWhenUnused());
		
		//创建该room的pipeline
		room.createPipelineForOuter();
		
		Room oldRoom = rooms.putIfAbsent(roomId, room);
		if (oldRoom != null) {
			log.warn("Room '{}' has just been created by another thread", roomId);
			return;
		}
		String kcName = "[NAME NOT AVAILABLE]";
		if (kurentoClient.getServerManager() != null) {
			kcName = kurentoClient.getServerManager().getName();
		}
		log.warn("No room '{}' exists yet. Created one " + "using KurentoClient '{}'.", roomId, kcName);
	}

	/**
	 * 关闭房间
	 * @param roomName
	 * @return 
	 * @throws RoomException
	 */
	public Set<UserParticipant> closeRoom(String roomName) throws RoomException {
		Room room = rooms.get(roomName);
		if (room == null) {
			throw new RoomException(Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
		}
		if (room.isClosed()) {
			throw new RoomException(Code.ROOM_CLOSED_ERROR_CODE, "Room '" + roomName + "' already closed");
		}
		Set<UserParticipant> participants = getParticipants(roomName);
		// copy the ids as they will be removed from the map
		Set<String> pids = new HashSet<String>(room.getParticipantIds());
		for (String pid : pids) {
			try {
				room.leave(pid);
			} catch (RoomException e) {
				log.warn("Error evicting participant with id '{}' from room '{}'", pid, roomName, e);
			}
		}
		room.close();
		rooms.remove(roomName);
		log.warn("Room '{}' removed and closed", roomName);
		return participants;
	}

	/**
	 * 获取用户所在的Pipeline
	 * @param participantId
	 * @return 
	 * @throws RoomException
	 */
	public MediaPipeline getPipeline(String participantId) throws RoomException {
		Participant participant = getParticipant(participantId);
		if (participant == null) {
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE,
					"No participant with id '" + participantId + "' was found");
		}
		return participant.getPipeline();
	}

	/**
	 * 获取用户所在房间的id
	 * @param participantId
	 * @return the name of the room
	 * @throws RoomException
	 */
	public String getRoomName(String participantId) throws RoomException {
		Participant participant = getParticipant(participantId);
		if (participant == null) {
			return null;
		}
		return participant.getRoom().getName();
	}

	/**
	 * 获取用户id
	 * @param participantId
	 * @return 
	 * @throws RoomException
	 */
	public String getParticipantName(String participantId) throws RoomException {
		Participant participant = getParticipant(participantId);
		return participant == null ? "" : participant.getName();
	}

	/**
	 * 获取用户ParticipantInfo
	 * @param participantId
	 * @return 
	 * @throws RoomException
	 */
	public UserParticipant getParticipantInfo(String participantId) throws RoomException {
		Participant participant = getParticipant(participantId);
		if(participant == null){
			return null;
		}
		return new UserParticipant(participantId, participant.getName());
	}

	private Participant getParticipant(String pid) {
		for (Room r : rooms.values()) {
			if (!r.isClosed()) {
				if (r.getParticipantIds().contains(pid) && r.getParticipant(pid) != null) {
					return r.getParticipant(pid);
				}
			}
		}
		return null;
	}

	/*
	 * 就RoomManager中的leaveRoom和disbandRoomByRoomName调用了
	 */
	class RoomCleanerTask extends java.util.TimerTask {
		String roomId = "";
		Timer timer = null;

		public RoomCleanerTask(Timer timer, String roomName) {
			this.timer = timer;
			this.roomId = roomName;
		}

		@Override
		public void run() {
			try {
				Room room = rooms.get(roomId);
				if (room != null) {
					Set<UserParticipant> remainingParticipants = null;
					try {
						remainingParticipants = getParticipants(roomId);
					} catch (RoomException e) {
						log.debug("Possible collision when closing the room '{}' (not found)");
						remainingParticipants = Collections.emptySet();
					}
					// 当房间内的成员第一次走光时,启动了定时器,定时期间,不管有没有人进入,只要定时器时间一到,此时,如果房间内没人,房间销毁(并不会刷新)
					if (remainingParticipants.isEmpty()) {
						log.info("方法:{}:{}","RoomCleanerTask.run()", "由于leaveRoom,定时时间到了,销毁房间");
						String roomName = room.getName();
						room.close();
						rooms.remove(roomName);

						/*
						 * add by wangjinglong delete server...
						 */
						KurentoClient client = room.GetKurentoClient();
						kcProvider.CloseServer(client);

						log.warn("Room '{}' removed and closed", roomName);
					}
				}
			} catch (RoomException re) {
				if (re.getCodeValue() == Code.ROOM_NOT_FOUND_ERROR_CODE.getValue()) {
					log.info("Room cleaner dismiss no exists room {} ", roomId);
				}
			}
			timer.cancel();
		}

	}

	public void recordRoom(String userName, String roomName) {
		Room room = rooms.get(roomName);
		if (room == null) {
			throw new RoomException(Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
		}
		if (room.isClosed()) {
			throw new RoomException(Code.ROOM_CLOSED_ERROR_CODE, "Room '" + roomName + "' already closed");
		}
		String fileName = RandomStringUtils.randomAlphanumeric(10);
		String filePath = RECORD_DIR + fileName;
		log.debug("Record file '{}' for room '{}'", filePath, roomName);
		room.record(userName, filePath);
	}

	public void stopRecordRoom(String userName, String roomName) {
		Room room = rooms.get(roomName);
		if (room == null) {
			throw new RoomException(Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
		}
		if (room.isClosed()) {
			throw new RoomException(Code.ROOM_CLOSED_ERROR_CODE, "Room '" + roomName + "' already closed");
		}
		room.stopRecord();
	}

	/**
	 * 根据roomName获取room对象
	 * @author wxiaohui
	 * @param roomName
	 * @return
	 */
	public Room getRoomByRoomName(String roomName) {
		Room room = rooms.get(roomName);
		return room;
	}

}
