package org.kurento.room.internal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.joda.time.DateTime;
import org.kurento.client.Composite;
import org.kurento.client.Continuation;
import org.kurento.client.ErrorEvent;
import org.kurento.client.EventListener;
import org.kurento.client.HubPort;
import org.kurento.client.IceCandidate;
import org.kurento.client.KurentoClient;
import org.kurento.client.MediaPipeline;
import org.kurento.client.MediaProfileSpecType;
import org.kurento.client.RecorderEndpoint;
import org.kurento.room.api.RoomHandler;
import org.kurento.room.exception.RoomException;
import org.kurento.room.exception.RoomException.Code;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSONObject;

/**
 * 房间对象
 */
public class Room {
	public static final int ASYNC_LATCH_TIMEOUT = 30;//异步超时锁
	private final static Logger log = LoggerFactory.getLogger(Room.class);
	private final ConcurrentMap<String, Participant> participants = new ConcurrentHashMap<String, Participant>();//所有的参与者数组
	private final String name;//房间名称
	private MediaPipeline pipeline;//房间在kms中对应的pipeline
	private CountDownLatch pipelineLatch = new CountDownLatch(1);//pipeline锁
	private KurentoClient kurentoClient;
	private RoomHandler roomHandler;
	private String owner;//房间创建者
	private boolean isRecording = false;//录屏状态
	private volatile boolean closed = false;//关闭状态
	private AtomicInteger activePublishers = new AtomicInteger(0);
	private Object pipelineCreateLock = new Object();
	private Object pipelineReleaseLock = new Object();
	private volatile boolean pipelineReleased = false;
	private boolean destroyKurentoClient;
	private List<Map<String, Object>> roomMeetingGroupDetails = new ArrayList<Map<String, Object>>();//房间详细资料
	private String roomLayoutStatus = "allvideo";//房间状态 allvideo-全视频状态、group-分组状态、whiteboard-白板状态、videoWhiteboard-白板加视频状态
	public Composite composite;
	private RecorderEndpoint recorderEndpoint;
	private HubPort hubPort;
	private long createTime;
	boolean hadMember;
	private String teacherInGroupId = "";//房间内老师所在的分组id
	private Map<String, Object> roomMeetingInfo = new HashMap<String, Object>();//房间会议信息
	
	/**
	 * 初始化对象
	 */
	public Room(String owner, String roomName, KurentoClient kurentoClient, RoomHandler roomHandler, boolean destroyKurentoClient) {
		this.name = roomName;
		this.kurentoClient = kurentoClient;
		this.destroyKurentoClient = destroyKurentoClient;
		this.roomHandler = roomHandler;
		this.owner = owner;
		this.createTime = new DateTime().getMillis();
		this.hadMember = false;
		log.debug("New ROOM instance, named '{}'", roomName);
	}
	
	/**
	 * 获取房主
	 */
	public String getOwner() {
		return owner;
	}

	/**
	 * 设置房间内分组的详细信息
	 */
	public void setRoomGroupDetails(List<Map<String, Object>> groupDetails) {
		this.roomMeetingGroupDetails = groupDetails;
	}
	
	/**
	 * 获取房间内分组的详细信息
	 */
	public List<Map<String, Object>> getRoomGroupDetails() {
		return this.roomMeetingGroupDetails;
	}

	/**
	 * 设置房间会议信息
	 */
	public void setRoomMeetingInfo(Map<String, Object> roomMeetingInfo) {
		this.roomMeetingInfo = roomMeetingInfo;
	}

	/**
	 * 获取房间会议信息
	 */
	public Map<String, Object> getRoomMeetingInfo() {
		return this.roomMeetingInfo;
	}

	/**
	 * 获取房间布局状态
	 */
	public String getRoomLayoutStatus() {
		return roomLayoutStatus;
	}

	/**
	 * 设置房间布局状态
	 */
	public void setRoomLayoutStatus(String roomLayoutStatus) {
		this.roomLayoutStatus = roomLayoutStatus;
	}

	/**
	 * 获取老师所在的分组
	 */
	public String getTeacherInGroupId() {
		return teacherInGroupId;
	}

	/**
	 * 获取老师所在的分组
	 */
	public void setTeacherInGroupId(String groupId) {
		this.teacherInGroupId = groupId;
	}

	/**
	 * used for get the kurento client
	 */
	public KurentoClient GetKurentoClient() {
		return kurentoClient;
	}

	/**
	 * 获取创建时间
	 */
	public long getCreateTime() {
		return this.createTime;
	}

	/**
	 * 这个属性没有setter 只是为那个全局定时器使用
	 */
	public boolean hadMember() {
		return this.hadMember;
	}

	/**
	 * 获取房间名称
	 */
	public String getName() {
		return name;
	}

	/**
	 * 获取房间的Pipeline
	 */
	public MediaPipeline getPipeline() {
		try {
			pipelineLatch.await(Room.ASYNC_LATCH_TIMEOUT, TimeUnit.SECONDS);
		} catch (InterruptedException e) {
			throw new RuntimeException(e);
		}
		return this.pipeline;
	}

	/**
	 * 加入房间
	 */
	public void join(String participantId, String userName, String role, boolean webParticipant) throws RoomException {
		checkClosed();
		if (userName == null || userName.isEmpty()) {
			throw new RoomException(Code.GENERIC_ERROR_CODE, "用户id为空");
		}
		for (Participant p : participants.values()) {
			if (p.getName().equals(userName)) {
				throw new RoomException(Code.EXISTING_USER_IN_ROOM_ERROR_CODE, "User '" + userName + "' 已经在房间 '" + name + "'");
			}
		}
//		createPipeline();
		Participant pt = new Participant(participantId, userName, this, getPipeline(), webParticipant);
		pt.setRole(role);
		participants.put(participantId, pt);
		if (isRecording) {
			pt.createRecordHubPort();
		}
		this.hadMember = true;
		log.info("ROOM {}: Added participant {}", name, userName);
	}
	
//	/**
//	 * 加入房间
//	 */
//	public void join2(String participantId, String userName, boolean webParticipant, String needRecord, String recordDir) throws RoomException {
//		checkClosed();
//		MediaProfileSpecType recordType = MediaProfileSpecType.MP4;
//		if (needRecord != null && needRecord.toLowerCase().indexOf("webm") > -1)
//			recordType = MediaProfileSpecType.WEBM;
//		if (userName == null || userName.isEmpty()) {
//			throw new RoomException(Code.GENERIC_ERROR_CODE, "Empty user name is not allowed");
//		}
//		for (Participant p : participants.values()) {
//			if (p.getName().equals(userName)) {
//				throw new RoomException(Code.EXISTING_USER_IN_ROOM_ERROR_CODE, "User '" + userName + "' already exists in room '" + name + "'");
//			}
//		}
//		createPipeline();
//		participants.put(participantId, new Participant(participantId, userName, this, getPipeline(), webParticipant));
//		// add recoding function
//		if (participants.size() == 1) {
//			// first user join
//			log.debug("ROOM {}: Start recording", getName());
//			this.hubPort = new HubPort.Builder(this.composite).build();
//			this.recorderEndpoint = new RecorderEndpoint.Builder(getPipeline(), recordDir).withMediaProfile(recordType).build();
//			log.debug("***********************************");
//			log.debug("******Record url is" + recordDir + "*********");
//			log.debug("*******" + recordType.toString() + "***********");
//			log.debug("********" + needRecord + "******");
//			log.debug("***********************************");
//			this.hubPort.connect(this.recorderEndpoint);
//			this.recorderEndpoint.record();
//		}
//
//		this.hadMember = true;
//
//		log.info("ROOM {}: Added participant {}", name, userName);
//	}

	public void newPublisher(Participant participant) {
		registerPublisher();
		// pre-load endpoints to recv video from the new publisher
		for (Participant participant1 : participants.values()) {
			if (participant.equals(participant1)) {
				continue;
			}
			participant1.getNewOrExistingSubscriber(participant.getName());
		}

		log.debug("ROOM {}: Virtually subscribed other participants {} to new publisher {}", name, participants.values(), participant.getName());
	}

	public void cancelPublisher(Participant participant) {
		deregisterPublisher();
		// cancel recv video from this publisher
		for (Participant subscriber : participants.values()) {
			if (participant.equals(subscriber)) {
				continue;
			}
			subscriber.cancelReceivingMedia(participant.getName());
		}

		log.debug("ROOM {}: Unsubscribed other participants {} from the publisher {}", name, participants.values(), participant.getName());
	}

	/**
	 * 离开房间
	 */
	public void leave(String participantId) throws RoomException {
		checkClosed();
		Participant participant = participants.get(participantId);
		if (participant == null) {
			throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE, "用户（" + participantId + "）不再房间 '" + name + "'中");
		}
		log.info("用户={}: 离开房间={}", participant.getName(), this.name);
		if (participant.isStreaming()) {
			this.deregisterPublisher();
		}
		this.removeParticipant(participant);
		participant.close();
	}

	/**
	 * 获取房间的所有的Participant
	 */
	public Collection<Participant> getParticipants() {
		checkClosed();
		return participants.values();
	}
	
	/**
	 * 获取房间内有多少个人
	 */
	public Integer getRoomMemberSize(){
		return participants.size();
	}
	
	/**
	 * 获取房间的所有的Participant的id
	 */
	public Set<String> getParticipantIds() {
		checkClosed();
		return participants.keySet();
	}

	/**
	 * 根据participantId获取房间内participant详情
	 */
	public Participant getParticipant(String participantId) {
		checkClosed();
		return participants.get(participantId);
	}

	/**
	 * 根据用户id获取Participant（没有返回null）
	 */
	public Participant getParticipantByName(String userName) {
		checkClosed();
		for (Participant p : participants.values()) {
			if (p.getName().equals(userName)) {
				return p;
			}
		}
		return null;
	}

	/**
	 * 获取房间内所有的userId
	 */
	public List<String> getRoomUserIds(){
		List<String> userIds = new ArrayList<>();
		for (Participant p : participants.values()) {
			userIds.add(p.getName());
		}
		return userIds;
	}
	
	/**
	 * 关闭房间
	 */
	public void close() {
		// stopRecord();
		if (!closed) {
			for (Participant user : participants.values()) {
				user.close();
			}
			participants.clear();
			closePipeline();
			log.debug("Room {} closed", this.name);
			if (destroyKurentoClient) {
				kurentoClient.destroy();
			}
			this.closed = true;
			this.updateClassStatus();
			
			long delRoomTime = System.currentTimeMillis();
			//房间的销毁时间
			long createRoomTime = this.createTime;
			long roomDurationl = delRoomTime - createRoomTime;
            log.info("room开始时间={},room关闭时间={},room时长",
            		new java.text.SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(new java.util.Date(createRoomTime)),
            		new java.text.SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(new java.util.Date(delRoomTime)), 
            		this.secondToTime(roomDurationl));
		} else {
			log.warn("Closing an already closed room '{}'", this.name);
		}
	}

	/**
	 * 更改课前房间状态,修改课程状态
	 */
	public void updateClassStatus(){
		Map<String, String> params = new HashMap<String, String>();
		params.put("roomId", this.name);
		params.put("status", "2");
		
		JSONObject result = HttpUtil.classPost(AutoUtil.getEnv().getProperty("classfront.url"), "updateLessonStatus", params);
		log.info("方法:{}中, 表达式{}的值: {}", "close", "result" , result);
	}
	
	public void sendIceCandidate(String participantId, String endpointName, IceCandidate candidate) {
		this.roomHandler.onIceCandidate(name, participantId, endpointName, candidate);
	}

	public void sendMediaError(String participantId, String description) {
		this.roomHandler.onMediaElementError(name, participantId, description);
	}

	/**
	 * 判断房间是否关闭
	 * @return
	 */
	public boolean isClosed() {
		return closed;
	}

	/**
	 * 检测房间是否关闭
	 */
	private void checkClosed() {
		if (closed) {
			throw new RoomException(Code.ROOM_CLOSED_ERROR_CODE, "The room '" + name + "' is closed");
		}
	}

	/**
	 * 删除Participant
	 */
	private void removeParticipant(Participant participant) {
		checkClosed();
		participants.remove(participant.getId());
		log.debug("删除房间={}中的用户id={}", this.name, participant.getName());
		for (Participant other : participants.values()) {
			other.cancelReceivingMedia(participant.getName());
		}
	}

	/**
	 * 获取订阅者个数
	 */
	public int getActivePublishers() {
		return activePublishers.get();
	}

	/**
	 * 注册订阅者
	 */
	public void registerPublisher() {
		this.activePublishers.incrementAndGet();
	}
	
	/**
	 * 撤销订阅者
	 */
	public void deregisterPublisher() {
		this.activePublishers.decrementAndGet();
	}
	
	/**
	 * 为外部创建Pipeline
	 */
	public void createPipelineForOuter() {
		this.createPipeline();
	}
	
	/**
	 * 创建Pipeline
	 */
	private void createPipeline() {
		synchronized (pipelineCreateLock) {
			if (pipeline != null) {
				return;
			}
			log.info("ROOM {}: Creating MediaPipeline", name);
			try {
				//Room类初始化时赋值
				kurentoClient.createMediaPipeline(new Continuation<MediaPipeline>() {
					@Override
					public void onSuccess(MediaPipeline result) throws Exception {
						pipeline = result;
						pipelineLatch.countDown();
						log.debug("ROOM {}: Created MediaPipeline", name);
					}

					@Override
					public void onError(Throwable cause) throws Exception {
						pipelineLatch.countDown();
						log.error("ROOM {}: Failed to create MediaPipeline", name, cause);
					}
				});
			} catch (Exception e) {
				log.error("Unable to create media pipeline for room '{}'", name, e);
				pipelineLatch.countDown();
			}
			if (getPipeline() == null) {
				throw new RoomException(Code.ROOM_CANNOT_BE_CREATED_ERROR_CODE,
						"Unable to create media pipeline for room '" + name + "'");
			}

			pipeline.addErrorListener(new EventListener<ErrorEvent>() {
				@Override
				public void onEvent(ErrorEvent event) {
					String desc = event.getType() + ": " + event.getDescription() + "(errCode=" + event.getErrorCode()
							+ ")";
					log.warn("ROOM {}: Pipeline error encountered: {}", name, desc);
					roomHandler.onPipelineError(name, getParticipantIds(), desc);
				}
			});
		}
	}

	/**
	 * 获取合流对象（刘浩写的）
	 */
	public Composite getComposite() {
		if (composite == null) {
			this.composite = new Composite.Builder(getPipeline()).build();
		}
		return composite;
	}

	/**
	 * 设置合流对象（刘浩写的）
	 */
	public void setComposite(Composite composite) {
		this.composite = composite;
	}

	/**
	 * 关闭Pipeline
	 */
	private void closePipeline() {
		synchronized (pipelineReleaseLock) {
			if (pipeline == null || pipelineReleased) {
				return;
			}
			stopRecord();
			getPipeline().release(new Continuation<Void>() {
				@Override
				public void onSuccess(Void result) throws Exception {
					log.debug("ROOM {}: Released Pipeline", Room.this.name);
					pipelineReleased = true;
				}
				@Override
				public void onError(Throwable cause) throws Exception {
					log.warn("ROOM {}: Could not successfully release Pipeline", Room.this.name, cause);
					pipelineReleased = true;
				}
			});
		}
	}

	/**
	 * 录制视频（刘浩写的）
	 */
	public void record(String userName, String filePath) {
		if (recorderEndpoint != null) return;
		for (Participant part : participants.values()) {
			part.createRecordHubPort();
		}
		MediaProfileSpecType recordType = MediaProfileSpecType.WEBM;
		this.hubPort = new HubPort.Builder(getComposite()).build();
		filePath = filePath + "." + recordType.name().toLowerCase();
		this.recorderEndpoint = new RecorderEndpoint.Builder(getPipeline(), filePath).withMediaProfile(recordType).build();
		log.debug("***********************************");
		log.debug("******Record url is {}!*********", filePath);
		log.debug("***********************************");
		this.hubPort.connect(this.recorderEndpoint);
		this.recorderEndpoint.record();
		isRecording = true;
	}

	/**
	 * 停止录屏（刘浩写的）
	 */
	public void stopRecord() {
		isRecording = false;
		if (composite != null) {
			try {
				composite.release();

			} catch (Exception e) {
				log.error("Room compsite Error calling release on elem #{} for {}", composite, name, e);
			}
			composite = null;
		}
		for (Participant part : participants.values()) {
			part.releaseRecordHubPort();
		}
		if (recorderEndpoint != null) {
			try {
				recorderEndpoint.stop();
				recorderEndpoint.release(new Continuation<Void>() {
					@Override
					public void onSuccess(Void result) throws Exception {
						log.debug("Room recorder Released successfully media element #{} for {}", recorderEndpoint,	name);
					}

					@Override
					public void onError(Throwable cause) throws Exception {
						log.warn("Room recorder Could not release media element #{} for {}", recorderEndpoint, name, cause);
					}
				});
			} catch (Exception e) {
				log.error("Room recorder Error calling release on elem #{} for {}", recorderEndpoint, name, e);
			}
			recorderEndpoint = null;
		}
		if (hubPort != null) {
			try {
				hubPort.release(new Continuation<Void>() {
					@Override
					public void onSuccess(Void result) throws Exception {
						log.debug("Room hubPort Released successfully media element #{} for {}", hubPort, name);
					}

					@Override
					public void onError(Throwable cause) throws Exception {
						log.warn("Room hubPort Could not release media element #{} for {}", hubPort, name, cause);
					}
				});
			} catch (Exception e) {
				log.error("Room hubPort Error calling release on elem #{} for {}", hubPort, name, e);
			}
			hubPort = null;
		}
	}

	/**
	 * 把数字转换成N天N小时N分N秒
	 */
    public String secondToTime(long second){
        long days = second / 86400;            //转换天数
        second = second % 86400;            //剩余秒数
        long hours = second / 3600;            //转换小时
        second = second % 3600;                //剩余秒数
        long minutes = second /60;            //转换分钟
        second = second % 60;                //剩余秒数
        if(days>0){
            return days + "天" + hours + "小时" + minutes + "分" + second + "秒";
        }else{
            return hours + "小时" + minutes + "分" + second + "秒";
        }
    }
}
