package org.kurento.room.rest;
import static org.kurento.commons.PropertiesManager.getProperty;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.kurento.room.NotificationRoomManager;
import org.kurento.room.RoomJsonRpcHandler;
import org.kurento.room.internal.Room;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

/**
 * 房间http请求接口
 * @author mabaojia
 * updateTime：2018年5月14日 10:20:23
 */
@RestController
public class RoomController {
	
	private static final Logger log = LoggerFactory.getLogger(RoomJsonRpcHandler.class);
	private static final int UPDATE_SPEAKER_INTERVAL_DEFAULT = 1800;
	private static final int THRESHOLD_SPEAKER_DEFAULT = -50;

	@Autowired
	private NotificationRoomManager roomManager;

	/**
	 * 创建房间
	 * @param uid	用户id
	 * @param roomNum	房间id
	 * @param roomName	房间名称
	 * @return
	 */
	@RequestMapping(value = "/rooms")
	public Map<String, Object> newRoom(@RequestParam Map<String, Object> data) {
		log.info("创建房间(post)，请求参数={}", data);
		Map<String, Object> map = new HashMap<String, Object>();
		
		String uid = data.get("uid").toString();
		String roomNum = data.get("roomNum").toString();
		String roomName = data.get("roomName").toString();
		
		if (StringUtils.isEmpty(uid) || StringUtils.isEmpty(roomNum)) {
			map.put("code", "40001");
			map.put("msg", "uid or roomid is null.");
			return map;
		}
		
		if(roomManager.isRoomExist(roomNum)){
			map.put("code", "-1");
			map.put("msg", "roomid is exist");
			return map;
		}

		roomManager.createRoom(uid, roomNum);
		log.info("用户={}请求创建房间roomNum={},roomName={},创建房间成功", uid, roomNum, roomName);
		
		map.put("code", "200");
		map.put("data", "");
		map.put("msg", "Create Room Is Success");
		return map;
	}
	
	/**
	 * 创建测试房间
	 */
	@RequestMapping("/createRoomTest")
	public Map<String,Object> createRoomTest(@RequestParam("userid") String userid, @RequestParam("roomname") String roomname , 
			@RequestParam("roomid") String roomid){
		roomManager.createRoom(userid, roomid);
		Map<String, Object> map = new HashMap<String, Object>();
		map.put("userid", userid);
		map.put("roomname", roomname);
		map.put("roomid", roomid);
		
		return map;
	}
	
	/**
	 * 获取当前房间信息
	 * @param roomid	房间id
	 * @return
	 */
	@RequestMapping("/queryRoomInfo")
	public Map<String, Object> queryRoomInfo(@RequestParam("roomid") String roomId){
		log.info("请求获取房间信息，roomid={}", roomId);
		Map<String, Object> resultMap = new HashMap<>();
		String roomLayoutStatus = roomManager.queryRoomLayoutStatus(roomId);
		String roomOwnerUserId = roomManager.queryRoomOwnerUserId(roomId);
		String roomPartsNum = "0";
		if(roomManager.getRoomManager().getRoomByRoomName(roomId) != null){
			roomPartsNum = String.valueOf(roomManager.getParticipants(roomId).size());	
		}
		
		resultMap.put("roomLayoutStatus", roomLayoutStatus);
		resultMap.put("roomOwnerUserId", roomOwnerUserId);
		resultMap.put("roomPartsNum", roomPartsNum);
		log.info("方法:{}中, 表达式{}的值: {}", "roomserver:queryRoomInfo", "resultMap" , resultMap);
		return resultMap;
	}
	
	/**
	 * 获取当前在直播的房间列表
	 * @return
	 */
	@RequestMapping("/queryOnGoingRoom")
	public Set<String> queryOnGoingRoom(){
		return roomManager.queryOnGoingRoom();
	}
	
	/**
	 * 销毁房间
	 * @param roomid	房间id
	 * @return
	 */
	@RequestMapping("/destroyRoom")
	public String destroyRoom(@RequestParam("roomid") String roomid) {
		//判断房间是否存在
		if (roomManager.getRoomManager().getRoomByRoomName(roomid) != null) {
			roomManager.getRoomManager().destoryRoom(roomid);
			return "销毁房间成功";
		}
		return "房间不存在";
	} 
	
	/**
	 * 判断房间是否存在
	 */
	@RequestMapping(value = "/rooms/check/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public @ResponseBody Map<String, Object> checkRoom(@PathVariable("id") String roomId) {
		Map<String, Object> map = new HashMap<String, Object>();
		if (StringUtils.isEmpty(roomId)) {
			map.put("code", "200");
			map.put("data", false);
			map.put("msg", "房间不存在");
			return map;
		}
		Room roomObj = roomManager.getRoomManager().getRoomByRoomName(roomId);
		map.put("code", "200");
		map.put("data", roomObj != null);
		return map;
	}

	/**
	 * 获取说话间隔
	 */
	@RequestMapping("/getUpdateSpeakerInterval")
	public Integer getUpdateSpeakerInterval() {
		return Integer.valueOf(getProperty("updateSpeakerInterval", UPDATE_SPEAKER_INTERVAL_DEFAULT));
	}

	/**
	 * 获取扬声器限制
	 */
	@RequestMapping("/getThresholdSpeaker")
	public Integer getThresholdSpeaker() {
		return Integer.valueOf(getProperty("thresholdSpeaker", THRESHOLD_SPEAKER_DEFAULT));
	}
}
