package org.kurento.room;
import java.util.Arrays;
import java.util.List;
import org.kurento.jsonrpc.DefaultJsonRpcHandler;
import org.kurento.jsonrpc.Session;
import org.kurento.jsonrpc.Transaction;
import org.kurento.jsonrpc.message.Request;
import org.kurento.room.api.pojo.ParticipantRequest;
import org.kurento.room.internal.ProtocolElements;
import org.kurento.room.rpc.JsonRpcNotificationService;
import org.kurento.room.rpc.JsonRpcUserControl;
import org.kurento.room.rpc.ParticipantSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import com.google.gson.JsonObject;

/**
 * @author Ivan Gracia (izanmail@gmail.com)
 * @author Micael Gallego (micael.gallego@gmail.com)
 * @since 1.0.0
 */
public class RoomJsonRpcHandler extends DefaultJsonRpcHandler<JsonObject> {
	
	private static final Logger log = LoggerFactory.getLogger(RoomJsonRpcHandler.class);

	private static final String HANDLER_THREAD_NAME = "handler";

	private JsonRpcUserControl userControl;

	private JsonRpcNotificationService notificationService;

	@Autowired
	public RoomJsonRpcHandler(JsonRpcUserControl userControl,
			JsonRpcNotificationService notificationService) {
		this.userControl = userControl;
		this.notificationService = notificationService;
	}

	@Override
	public List<String> allowedOrigins() {
		return Arrays.asList("*");
	}

	@Override
	public final void handleRequest(Transaction transaction, Request<JsonObject> request)throws Exception {
		
		String sessionId = null;
		try {
			sessionId = transaction.getSession().getSessionId();
		} catch (Throwable e) {
			log.warn("Error getting session id from transaction {}", transaction, e);
			throw e;
		}

		updateThreadName(HANDLER_THREAD_NAME + "_" + sessionId);

		log.debug("Session #{} - request: {}", sessionId, request);

		notificationService.addTransaction(transaction, request);

		ParticipantRequest participantRequest = new ParticipantRequest(sessionId, Integer.toString(request.getId()));

		transaction.startAsync();

		switch (request.getMethod()) {
			//加入房间
			case ProtocolElements.JOINROOM_METHOD :
				userControl.joinRoom(transaction, request, participantRequest);
				break;
			//推流
			case ProtocolElements.PUBLISHVIDEO_METHOD :
				userControl.publishVideo(transaction, request, participantRequest);
				break;
			//录制
			case ProtocolElements.RECORDVIDEO_METHOD :
				userControl.recordVideo(transaction, request, participantRequest);
				break;
			//停止录制
			case ProtocolElements.RECORDSTOP_METHOD :
				userControl.stopRecordVideo(transaction, request, participantRequest);
				break;
			//停止推流
			case ProtocolElements.UNPUBLISHVIDEO_METHOD :
				userControl.unpublishVideo(transaction, request, participantRequest);
				break;
			//订阅视频流
			case ProtocolElements.RECEIVEVIDEO_METHOD :
				userControl.receiveVideoFrom(transaction, request, participantRequest);
				break;
			//退订视频流
			case ProtocolElements.UNSUBSCRIBEFROMVIDEO_METHOD :
				userControl.unsubscribeFromVideo(transaction, request, participantRequest);
				break;
			//交换信令
			case ProtocolElements.ONICECANDIDATE_METHOD :
				userControl.onIceCandidate(transaction, request, participantRequest);
				break;
			//离开房间
			case ProtocolElements.LEAVEROOM_METHOD :
				userControl.leaveRoom(transaction, request, participantRequest);
				break;
			//发送消息
			case ProtocolElements.SENDMESSAGE_ROOM_METHOD :
    	  		userControl.sendMessage(transaction, request, participantRequest);
    	  		break;
    	  	//自定义请求
			case ProtocolElements.CUSTOMREQUEST_METHOD :
				userControl.customRequest(transaction, request, participantRequest);
				break;
			//创建分组和修改分组(其他人)
			case ProtocolElements.CREATE_METTING_GROUP_METHOD :
				userControl.createMettingGroup(transaction, request, participantRequest);
				break;
			//解散分组(其他人)
			case ProtocolElements.DISSOLUTION_METTING_GROUP_METHOD :
				userControl.dissolutionMettingGroup(transaction, request, participantRequest);
				break;
			//删除分组(不通知)
			case ProtocolElements.DEL_METTING_GROUP_METHOD :
				userControl.delMettingGroup(transaction, request, participantRequest);
				break;
			//老师进入分组(其他人)
			case ProtocolElements.TEACHER_GOING_GROUP_METHOD :
				userControl.teacherGoingGroup(transaction, request, participantRequest);
				break;
			//解散房间(其他人)
			case ProtocolElements.DISBAND_ROOM_METHOD:
				userControl.disbandRoom(transaction, request, participantRequest);
				break;
			//更改房间布局状态(其他人)
			case ProtocolElements.UPDATE_ROOM_STATUS_METHOD:
				userControl.updateRoomStatus(transaction, request, participantRequest);
				break;
			//老师发送问题(其他人)
			case ProtocolElements.SEND_QUESTION_METHOD:
				userControl.sendQuestion(transaction, request, participantRequest);
				break;
			//答题结束统计结果(其他人)
			case ProtocolElements.SEND_ANSWER_QUESTION_RESULT_METHOD:
				userControl.sendAnswerQuestionResult(transaction, request, participantRequest);
				break;
			//老师关闭答题卡(其他人)
			case ProtocolElements.CLOSE_ANSWER_SHEET_METHOD:
				userControl.closeAnswerSheet(transaction, request, participantRequest);
				break;
			//老师一键操作(一键静音和一键复位)(其他人)
			case ProtocolElements.ONE_KEY_OPERATION_METHOD:
				userControl.oneKeyOperation(transaction, request, participantRequest);
				break;
			//举手(其他人)
			case ProtocolElements.RAISE_HANDS_METHOD:
				userControl.raiseHands(transaction, request, participantRequest);
				break;
			//设置个人流状态(个人)
			case ProtocolElements.SET_PEOPLE_VIDEO_STATUS_METHOD :
				userControl.setPeopleVideoStatus(transaction, request, participantRequest);
				break;
			//站起来(个人)
			case ProtocolElements.ROLL_CALL_METHOD :
				userControl.rollCall(transaction, request, participantRequest);
				break;
			//坐回去(个人)
			case ProtocolElements.SIT_DOWN_METHOD:
				userControl.sitDown(transaction, request, participantRequest);
				break;
			//开启或关闭某一个人的音频(个人)
			case ProtocolElements.TOGGLE_SOMEONE_AUDIO_METHOD:
				userControl.toggleSomeoneAudio(transaction, request, participantRequest);
				break;
			//老师开启或关闭某个学生的视频,具体开启关闭,通过传参数,然后前端自己判断(个人)
			case ProtocolElements.TOGGLE_SOMEONE_VIDEO_METHOD:
				userControl.toggleSomeoneVideo(transaction, request, participantRequest);
				break;
			//授权学生操作白板(个人)
			case ProtocolElements.ALLOW_OPERATE_METHOD:
				userControl.allowOperate(transaction, request, participantRequest);
				break;
			//学生给老师发送答案(个人)
			case ProtocolElements.SEND_ANSWER_METHOD:
				userControl.sendAnswer(transaction, request, participantRequest);
				break;
			default :
				log.error("Unrecognized request {}", request);
				break;
		}
		updateThreadName(HANDLER_THREAD_NAME);
	}

	@Override
	public final void afterConnectionClosed(Session session, String status) throws Exception {
		ParticipantSession ps = null;
		if (session.getAttributes().containsKey(ParticipantSession.SESSION_KEY)) {
			ps = (ParticipantSession) session.getAttributes().get(ParticipantSession.SESSION_KEY);
		}
		String sid = session.getSessionId();
		log.debug("CONN_CLOSED: sessionId={}, participant in session: {}", sid, ps);
		ParticipantRequest preq = new ParticipantRequest(sid, null);
		updateThreadName(sid + "|wsclosed");
		userControl.leaveRoom(null, null, preq);
		updateThreadName(HANDLER_THREAD_NAME);
	}

	@Override
	public void handleTransportError(Session session, Throwable exception) throws Exception {
		log.debug("Transport error for session id {}", session != null ? session.getSessionId() : "NULL_SESSION", exception);
	}

	private void updateThreadName(String name) {
		Thread.currentThread().setName("user:" + name);
	}
}
