package org.kurento.room.rpc;
import java.io.IOException;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import org.kurento.jsonrpc.Session;
import org.kurento.jsonrpc.Transaction;
import org.kurento.jsonrpc.message.Request;
import org.kurento.room.NotificationRoomManager;
import org.kurento.room.api.pojo.ParticipantRequest;
import org.kurento.room.api.pojo.UserParticipant;
import org.kurento.room.exception.RoomException;
import org.kurento.room.internal.ProtocolElements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.gson.JsonObject;

/**
 * Controls the user interactions by delegating her JSON-RPC requests to the room API.
 */
public class JsonRpcUserControl {

	private static final Logger log = LoggerFactory.getLogger(JsonRpcUserControl.class);
//	private static final int SCHEDULE_TIME = 3 * 60000;
//	private final int DEFAULT_MEMBER_LIMIT = PropertiesManager.getProperty("app.memberLimit", 6);
	protected NotificationRoomManager roomManager;

	@Autowired
	public JsonRpcUserControl(NotificationRoomManager roomManager) {
		this.roomManager = roomManager;
	}

	/**
	 * 加入房间
	 */
	public void joinRoom(Transaction transaction, Request<JsonObject> request, ParticipantRequest participantRequest)
			throws IOException, InterruptedException, ExecutionException {
		String roomName = getStringParam(request, ProtocolElements.JOINROOM_ROOM_PARAM);
		String userName = getStringParam(request, ProtocolElements.JOINROOM_USER_PARAM);
		String role = "normal";

		boolean bRole = hasRole(request, ProtocolElements.PARTICIPANTPUBLISHED_USER_ROLE);
		if (bRole) {
			role = getStringParam(request, ProtocolElements.PARTICIPANTPUBLISHED_USER_ROLE);
		}

		ParticipantSession participantSession = getParticipantSession(transaction);
		participantSession.setParticipantName(userName);
		participantSession.setRoomName(roomName);

		roomManager.joinRoom(userName, role, roomName, true, participantRequest);
	}

	/**
	 * 客户端推流操作
	 */
	public void publishVideo(Transaction transaction, Request<JsonObject> request, ParticipantRequest participantRequest) {
		String sdpOffer = getStringParam(request, ProtocolElements.PUBLISHVIDEO_SDPOFFER_PARAM);
		boolean doLoopback = getBooleanParam(request, ProtocolElements.PUBLISHVIDEO_DOLOOPBACK_PARAM);
		String userId = getStringParam(request, ProtocolElements.USER_ID, "");
		
		roomManager.publishMedia(participantRequest, sdpOffer, doLoopback, userId);
	}

	public void unpublishVideo(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) {
		log.info("进入{}方法了!!!!!", "unpublishVideo");
		roomManager.unpublishMedia(participantRequest);
	}

	/**
	 * 订阅视频流
	 * @param rateLevel		1-低码率2-中码率3-高码率4-不变化码率（默认4）
	 */
	public void receiveVideoFrom(final Transaction transaction, final Request<JsonObject> request,
			ParticipantRequest participantRequest) {
		log.info("进入{}方法了!!!!!", "receiveVideoFrom");
		
		Integer rateLevel = getIntParam(request, "rateLevel", 4);
		String senderName = getStringParam(request, ProtocolElements.RECEIVEVIDEO_SENDER_PARAM);
		senderName = senderName.substring(0, senderName.indexOf("_"));
		String sdpOffer = getStringParam(request, ProtocolElements.RECEIVEVIDEO_SDPOFFER_PARAM);

		roomManager.subscribe(senderName, sdpOffer, participantRequest, rateLevel);
	}

	public void unsubscribeFromVideo(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) {
		log.info("进入{}方法了!!!!!", "unsubscribeFromVideo");
		String senderName = getStringParam(request, ProtocolElements.UNSUBSCRIBEFROMVIDEO_SENDER_PARAM);
		senderName = senderName.substring(0, senderName.indexOf("_"));

		roomManager.unsubscribe(senderName, participantRequest);
	}

	public void leaveRoomAfterConnClosed(String sessionId) {
		try {
			roomManager.evictParticipant(sessionId);
			log.info("Evicted participant with sessionId {}", sessionId);
		} catch (RoomException e) {
			log.warn("Unable to evict: {}", e.getMessage());
			log.trace("Unable to evict user", e);
		}
	}

	public void leaveRoom(Transaction transaction, Request<JsonObject> request, ParticipantRequest participantRequest) {
		boolean exists = false;
		String pid = participantRequest.getParticipantId();
		// trying with room info from session
		String roomName = null;
		if (transaction != null) {
			roomName = getParticipantSession(transaction).getRoomName();
		}
		if (roomName == null) { // null when afterConnectionClosed
			log.warn("No room information found for participant with session Id {}. "
					+ "Using the admin method to evict the user.", pid);
			leaveRoomAfterConnClosed(pid);
		} else {
			// sanity check, don't call leaveRoom unless the id checks out
			Set<UserParticipant> roomParts = roomManager.getParticipants(roomName);
			for (UserParticipant part : roomParts) {
 				if (part.getParticipantId().equals(participantRequest.getParticipantId())) {
					exists = true;
					break;
				}
			}

			if (exists) {
				log.debug("Participant with sessionId {} is leaving room {}", pid, roomName);
				roomManager.leaveRoom(participantRequest);
				log.info("Participant with sessionId {} has left room {}", pid, roomName);
			} else {
				log.warn("Participant with session Id {} not found in room {}. "
						+ "Using the admin method to evict the user.", pid, roomName);
				leaveRoomAfterConnClosed(pid);
			}
		}
	}

	public void onIceCandidate(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) {
		log.info("进入{}方法了!!!!!", "onIceCandidate");
		String endpointName = getStringParam(request, ProtocolElements.ONICECANDIDATE_EPNAME_PARAM);
		String candidate = getStringParam(request, ProtocolElements.ONICECANDIDATE_CANDIDATE_PARAM);
		String sdpMid = getStringParam(request, ProtocolElements.ONICECANDIDATE_SDPMIDPARAM);
		int sdpMLineIndex = getIntParam(request, ProtocolElements.ONICECANDIDATE_SDPMLINEINDEX_PARAM);

		roomManager.onIceCandidate(endpointName, candidate, sdpMLineIndex, sdpMid, participantRequest);
	}

	public void sendMessage(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) {
		log.info("进入{}方法了!!!!!", "sendMessage");
		String userName = getStringParam(request, ProtocolElements.SENDMESSAGE_USER_PARAM);
		String roomName = getStringParam(request, ProtocolElements.SENDMESSAGE_ROOM_PARAM);
		String message = getStringParam(request, ProtocolElements.SENDMESSAGE_MESSAGE_PARAM);

		log.debug("Message from {} in room {}: '{}'", userName, roomName, message);

		roomManager.sendMessage(message, userName, roomName, participantRequest);
	}

	public void customRequest(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) {
		throw new RuntimeException("Unsupported method");
	}

	public ParticipantSession getParticipantSession(Transaction transaction) {
		Session session = transaction.getSession();
		ParticipantSession participantSession = (ParticipantSession) session.getAttributes()
				.get(ParticipantSession.SESSION_KEY);
		if (participantSession == null) {
			participantSession = new ParticipantSession();
			session.getAttributes().put(ParticipantSession.SESSION_KEY, participantSession);
		}
		return participantSession;
	}

	public static boolean hasRole(Request<JsonObject> request, String key) {
		if (request.getParams() == null || request.getParams().get(key) == null) {
			return false;
		}

		return true;
	}

	public static String getStringParam(Request<JsonObject> request, String key) {
		if (request.getParams() == null || request.getParams().get(key) == null) {
			throw new RuntimeException("Request element '" + key + "' is missing");
		}
		return request.getParams().get(key).getAsString();
	}
	
	public static String getStringParam(Request<JsonObject> request, String key, String defaultParam) {
		if (request.getParams() == null || request.getParams().get(key) == null) {
			return defaultParam;
		}
		return request.getParams().get(key).getAsString();
	}
	
	public static Integer getIntParam(Request<JsonObject> request, String key, Integer defaultParam) {
		if (request.getParams() == null || request.getParams().get(key) == null) {
			return defaultParam;
		}
		return request.getParams().get(key).getAsInt();
	}

	public static int getIntParam(Request<JsonObject> request, String key) {
		if (request.getParams() == null || request.getParams().get(key) == null) {
			throw new RuntimeException("Request element '" + key + "' is missing");
		}
		return request.getParams().get(key).getAsInt();
	}

	public static boolean getBooleanParam(Request<JsonObject> request, String key) {
		if (request.getParams() == null || request.getParams().get(key) == null) {
			throw new RuntimeException("Request element '" + key + "' is missing");
		}
		return request.getParams().get(key).getAsBoolean();
	}

	public void recordVideo(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) {
		String roomName = getStringParam(request, ProtocolElements.JOINROOM_ROOM_PARAM);
		String userName = getStringParam(request, ProtocolElements.JOINROOM_USER_PARAM);

		ParticipantSession participantSession = getParticipantSession(transaction);
		participantSession.setParticipantName(userName);
		participantSession.setRoomName(roomName);

		roomManager.recordRoom(userName, roomName, participantRequest);
	}

	public void stopRecordVideo(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) {
		String roomName = getStringParam(request, ProtocolElements.JOINROOM_ROOM_PARAM);
		String userName = getStringParam(request, ProtocolElements.JOINROOM_USER_PARAM);

		ParticipantSession participantSession = getParticipantSession(transaction);
		participantSession.setParticipantName(userName);
		participantSession.setRoomName(roomName);

		roomManager.stopRecordRoom(userName, roomName, participantRequest);
	}

	/**
	 * 创建分组
	 */
	public void createMettingGroup(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		log.info("进入{}方法了!!!!!", "createMettingGroup");
		// 获取从前端传过来的参数
		String userId = getStringParam(request, ProtocolElements.USER_ID);
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);
		String classId = getStringParam(request, ProtocolElements.CLASS_ID);
		String classMettingId = getStringParam(request, ProtocolElements.CLASS_METTING_ID);
		String cmName = getStringParam(request, ProtocolElements.CM_NAME);
		String number = getStringParam(request, ProtocolElements.NUMBER);
		String groupInfo = getStringParam(request, ProtocolElements.GROUP_INFO);

		roomManager.createMettingGroup(userId, roomName, classMettingId, classId, cmName, number, groupInfo,
				participantRequest);
	}

	/**
	 * 解散会议
	 */
	public void dissolutionMettingGroup(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		log.info("进入{}方法了!!!!!", "dissolutionMettingGroup");
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);

		roomManager.dissolutionMettingGroup(roomName, participantRequest);
	}

	/**
	 * 删除会议
	 */
	public void delMettingGroup(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		log.info("进入{}方法了!!!!!", "delMettingGroup");
		String classMettingId = getStringParam(request, ProtocolElements.CLASS_METTING_ID);
		String roomId = getStringParam(request, ProtocolElements.ROOM_ID);
		String userId = getStringParam(request, ProtocolElements.USER_ID);

		roomManager.delMettingGroup(userId, roomId, classMettingId, participantRequest);
	}

	/**
	 * 老师进入小组
	 */
	public void teacherGoingGroup(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		log.info("进入{}方法了!!!!!", "teacherGoingGroup");
		// 获取从前端传过来的参数
		String groupId = getStringParam(request, ProtocolElements.GROUP_ID);
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);
		String status = getStringParam(request, ProtocolElements.STATUS);

		roomManager.teacherGoingGroup(status, roomName, groupId, participantRequest);
	}

	/**
	 * 解散房间
	 */
	public void disbandRoom(Transaction transaction, Request<JsonObject> request, ParticipantRequest participantRequest)
			throws IOException, InterruptedException, ExecutionException {
		log.info("进入{}方法了!!!!!", "disbandRoom");
		// 获取从前端传过来的参数
		JsonObject params = request.getParams();
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);
		roomManager.disbandRoom(params, roomName, participantRequest);
	}

	/**
	 * 更改房间布局状态
	 */
	public void updateRoomStatus(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		log.info("进入{}方法了!!!!!", "updateRoomStatus");
		// 获取从前端传过来的参数
		JsonObject params = request.getParams();
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);
		String roomLayoutStatus = getStringParam(request, ProtocolElements.LAYOUT_ROOM_STATUS);
		roomManager.updateRoomStatus(participantRequest, params, roomName, roomLayoutStatus);
	}

	/**
	 * 老师发送问题
	 */
	public void sendQuestion(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		log.info("进入{}方法了!!!!!", "sendQuestion");
		// 获取从前端传过来的参数
		JsonObject params = request.getParams();
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);
		roomManager.sendQuestion(params, roomName, participantRequest);
	}

	/**
	 * 答题结束统计结果
	 */
	public void sendAnswerQuestionResult(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		log.info("进入{}方法了!!!!!", "sendAnswerQuestionResult");
		JsonObject params = request.getParams();
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);

		roomManager.sendAnswerQuestionResult(params, roomName, participantRequest);
	}

	/**
	 * 老师关闭答题卡
	 */
	public void closeAnswerSheet(Transaction transaction, Request<JsonObject> request,

			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		JsonObject params = request.getParams();
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);

		roomManager.closeAnswerSheet(params, roomName, participantRequest);
	}

	/**
	 * 老师一键操作
	 */
	public void oneKeyOperation(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		JsonObject params = request.getParams();
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);

		roomManager.oneKeyOperation(params, roomName, participantRequest);
	}

	/**
	 * 举手
	 */
	public void raiseHands(Transaction transaction, Request<JsonObject> request, ParticipantRequest participantRequest)
			throws IOException, InterruptedException, ExecutionException {
		JsonObject params = request.getParams();
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);

		roomManager.raiseHands(params, roomName, participantRequest);
	}

	/**
	 * 设置个人流状态
	 */
	public void setPeopleVideoStatus(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) {
		String userId = getStringParam(request, ProtocolElements.USER_ID);
		String videoStatus = getStringParam(request, ProtocolElements.VIDEO_STATUS);
		log.debug("setPeopleVideoStatus to {}", userId);
		roomManager.setPeopleVideoStatus(userId, videoStatus, participantRequest);
	}

	/**
	 * 站起来
	 */
	public void rollCall(Transaction transaction, Request<JsonObject> request, ParticipantRequest participantRequest) {
		String userName = getStringParam(request, ProtocolElements.ROLL_CALL_USER);
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);
		String roomLayoutStatus = getStringParam(request, ProtocolElements.LAYOUT_ROOM_STATUS);

		log.debug("RollCall to {}", userName);
		roomManager.rollCall(userName, roomName, roomLayoutStatus, participantRequest);
	}

	/**
	 * 做下
	 */
	public void sitDown(Transaction transaction, Request<JsonObject> request, ParticipantRequest participantRequest) {
		String sitDownUser = getStringParam(request, ProtocolElements.SIT_DOWN_USER);
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);
		String roomLayoutStatus = getStringParam(request, ProtocolElements.LAYOUT_ROOM_STATUS);

		log.debug("sitDown to {}", sitDownUser);
		roomManager.sitDown(sitDownUser, roomName, roomLayoutStatus, participantRequest);
	}

	/**
	 * 开启或关闭某一个人的音频
	 */
	public void toggleSomeoneAudio(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		JsonObject params = request.getParams();
		String userName = getStringParam(request, ProtocolElements.USER_ID);
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);

		roomManager.toggleSomeoneAudio(params, userName, roomName, participantRequest);
	}

	/**
	 * 老师开启或关闭某个学生的视频
	 */
	public void toggleSomeoneVideo(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		JsonObject params = request.getParams();
		String userName = getStringParam(request, ProtocolElements.USER_ID);
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);

		roomManager.toggleSomeoneVideo(params, userName, roomName, participantRequest);
	}

	/**
	 * 授权学生操作白板
	 */
	public void allowOperate(Transaction transaction, Request<JsonObject> request,
			ParticipantRequest participantRequest) throws IOException, InterruptedException, ExecutionException {
		JsonObject params = request.getParams();
		String userName = getStringParam(request, ProtocolElements.USER_ID);
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);

		roomManager.allowOperate(params, userName, roomName, participantRequest);
	}

	/**
	 * 学生给老师发送答案
	 */
	public void sendAnswer(Transaction transaction, Request<JsonObject> request, ParticipantRequest participantRequest)
			throws IOException, InterruptedException, ExecutionException {
		JsonObject params = request.getParams();
		String userName = getStringParam(request, ProtocolElements.USER_ID);
		String roomName = getStringParam(request, ProtocolElements.ROOM_ID);

		roomManager.sendAnswer(params, userName, roomName, participantRequest);
	}

}
