package org.kurento.room;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.annotation.PreDestroy;
import org.kurento.client.MediaElement;
import org.kurento.client.MediaPipeline;
import org.kurento.client.MediaType;
import org.kurento.room.api.KurentoClientProvider;
import org.kurento.room.api.KurentoClientSessionInfo;
import org.kurento.room.api.MutedMediaType;
import org.kurento.room.api.NotificationRoomHandler;
import org.kurento.room.api.UserNotificationService;
import org.kurento.room.api.pojo.ParticipantRequest;
import org.kurento.room.api.pojo.UserParticipant;
import org.kurento.room.exception.RoomException;
import org.kurento.room.exception.RoomException.Code;
import org.kurento.room.internal.DefaultKurentoClientSessionInfo;
import org.kurento.room.internal.DefaultNotificationRoomHandler;
import org.kurento.room.internal.HttpUtil;
import org.kurento.room.internal.ProtocolElements;
import org.kurento.room.internal.Room;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.gson.JsonObject;

/**
 * 房间管理
 * （针对用户发起的http接口）
 * @author mabaojia
 */
public class NotificationRoomManager {
	@Autowired
	private Environment env;
	private final Logger log = LoggerFactory.getLogger(NotificationRoomManager.class);
	private NotificationRoomHandler notificationRoomHandler;
	private RoomManager internalManager;

	/**
	 * Provides an instance of the room manager by setting an user notification
	 * service that will be used by the default event handler to send responses
	 * and notifications back to the clients.
	 *
	 * @param notificationService
	 *            encapsulates the communication layer, used to instantiate
	 *            {@link DefaultNotificationRoomHandler}
	 * @param kcProvider
	 *            enables the manager to obtain Kurento Client instances
	 */
	public NotificationRoomManager(UserNotificationService notificationService, KurentoClientProvider kcProvider) {
		super();
		this.notificationRoomHandler = new DefaultNotificationRoomHandler(notificationService);
		this.internalManager = new RoomManager(notificationRoomHandler, kcProvider);
	}

	/**
	 * Provides an instance of the room manager by setting an event handler.
	 *
	 * @param notificationRoomHandler
	 *            the room event handler implementation
	 * @param kcProvider
	 *            enables the manager to obtain Kurento Client instances
	 */
	public NotificationRoomManager(NotificationRoomHandler notificationRoomHandler, KurentoClientProvider kcProvider) {
		super();
		this.notificationRoomHandler = notificationRoomHandler;
		this.internalManager = new RoomManager(notificationRoomHandler, kcProvider);
	}

	// ----------------- CLIENT-ORIGINATED REQUESTS ------------

	/**
	 * Calls
	 * {@link RoomManager#joinRoom(String, String, boolean, KurentoClientSessionInfo, String)}
	 * with a {@link DefaultKurentoClientSessionInfo} bean as implementation of
	 * @param request
	 * @param needRecord 
	 * @param recordDir 
	 * @see RoomManager#joinRoom(String, String, boolean,
	 *      KurentoClientSessionInfo, String)
	 */
	public void joinRoom(String userName, String role, String roomName, boolean webParticipant, ParticipantRequest request) {
		Set<UserParticipant> existingParticipants = null;
		Map<String,Object> roomMeetingInfo = new HashMap<String, Object>();
		String roomLayoutStatus = "";
		List<Map<String,Object>> roomGroupDetails = new ArrayList<Map<String, Object>>();
		String teacherInGroupId = "";
		String ownerUser = "";
		try {
			KurentoClientSessionInfo kcSessionInfo = new DefaultKurentoClientSessionInfo(request.getParticipantId(),
					roomName);
			existingParticipants = internalManager.joinRoom(userName, role, roomName, webParticipant, kcSessionInfo,
					request.getParticipantId());
			
			Room room = internalManager.getRoomByRoomName(roomName);
			roomMeetingInfo = room.getRoomMeetingInfo();
			roomLayoutStatus = room.getRoomLayoutStatus();
			roomGroupDetails = room.getRoomGroupDetails();
			teacherInGroupId = room.getTeacherInGroupId();
			ownerUser = room.getOwner();
			
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error joining/creating room {}", userName, roomName, e);
			notificationRoomHandler.onParticipantJoined(request, roomName, ownerUser, userName, role, null, e, roomMeetingInfo , roomLayoutStatus, teacherInGroupId,
					roomGroupDetails);
		}
		if (existingParticipants != null) {
			notificationRoomHandler.onParticipantJoined(request, roomName ,ownerUser, userName, role, existingParticipants, null, roomMeetingInfo , roomLayoutStatus, teacherInGroupId,
					roomGroupDetails);
		}
	}
	
	/**
	 * @param request
	 *            instance of {@link ParticipantRequest} POJO
	 *
	 * @see RoomManager#leaveRoom(String)
	 */
	public void leaveRoom(ParticipantRequest request) {
		String pid = request.getParticipantId();
		Set<UserParticipant> remainingParticipants = null;
		String roomName = null;
		String userName = null;
		try {
			roomName = internalManager.getRoomName(pid);
			userName = internalManager.getParticipantName(pid);
			remainingParticipants = internalManager.leaveRoom(pid);
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error leaving room {}", userName, roomName, e);
			notificationRoomHandler.onParticipantLeft(request, null, null, e);
		}
		if (remainingParticipants != null) {
			notificationRoomHandler.onParticipantLeft(request, userName, remainingParticipants, null);
		}
	}

	/**
	 * @param request
	 *            instance of {@link ParticipantRequest} POJO
	 *
	 * @see RoomManager#publishMedia(String, boolean, String, MediaElement,
	 *      MediaType, boolean, MediaElement...)
	 */
	public void publishMedia(ParticipantRequest request, boolean isOffer, String sdp,MediaElement loopbackAlternativeSrc, 
			MediaType loopbackConnectionType, boolean doLoopback, String userId, MediaElement... mediaElements) {
		String pid = request.getParticipantId();
		String userName = null;
		Set<UserParticipant> participants = null;
		String sdpAnswer = null;
		String role = null;
		String roomStatus = null;
		String teacherGroupId = "";
		try {
			userName = internalManager.getParticipantName(pid);
			
			sdpAnswer = internalManager.publishMedia(request.getParticipantId(), isOffer, sdp, loopbackAlternativeSrc,
					loopbackConnectionType, doLoopback, mediaElements);
			
			participants = internalManager.getParticipants(internalManager.getRoomName(pid));
			role = internalManager.getParticipantRole(pid);
			roomStatus = internalManager.getParticipantVideoStatus(pid);
			teacherGroupId = internalManager.getTeacherGroupId(pid);
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error publishing media", userName, e);
			notificationRoomHandler.onPublishMedia(request, null, null, null, null, null, null, userId, e);
		}
		if (sdpAnswer != null) {
			notificationRoomHandler.onPublishMedia(request, userName, role, roomStatus, teacherGroupId, sdpAnswer, participants, userId, null);
		}
	}

	/**
	 * 推流
	 */
	public void publishMedia(ParticipantRequest request, String sdpOffer, boolean doLoopback, String userId,
			MediaElement... mediaElements) {
		this.publishMedia(request, true, sdpOffer, null, null, doLoopback, userId, mediaElements);
	}

	/**
	 * @param request
	 *            instance of {@link ParticipantRequest} POJO
	 *
	 * @see RoomManager#unpublishMedia(String)
	 */
	public void unpublishMedia(ParticipantRequest request) {
		String pid = request.getParticipantId();
		String userName = null;
		Set<UserParticipant> participants = null;
		boolean unpublished = false;
		try {
			userName = internalManager.getParticipantName(pid);
			internalManager.unpublishMedia(pid);
			unpublished = true;
			participants = internalManager.getParticipants(internalManager.getRoomName(pid));
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error unpublishing media", userName, e);
			notificationRoomHandler.onUnpublishMedia(request, null, null, e);
		}
		if (unpublished) {
			notificationRoomHandler.onUnpublishMedia(request, userName, participants, null);
		}
	}

	/**
	 * @param request
	 *            instance of {@link ParticipantRequest} POJO
	 * @see RoomManager#subscribe(String, String, String)
	 * @param remoteName	被订阅者uid
	 * @param sdpOffer		订阅者的offer
	 * @param request		请求体
	 */
	public void subscribe(String remoteName, String sdpOffer, ParticipantRequest request, Integer rateLevel) {
		//用户sessionId
		String pid = request.getParticipantId();
		String userName = null;
		String sdpAnswer = null;
		try {
			//根据订阅者sessionId获取订阅者userId
			userName = internalManager.getParticipantName(pid);
			//根据pid获取房间id，根据房间id获取房间对象
//			Room room = internalManager.getRoomByRoomName(internalManager.getRoomName(pid));
//			
//			//获取房主id
//			String ownerUserId = room.getOwner();
//			Integer roomUserSize = room.getRoomMemberSize();
//			if(remoteName.equals(ownerUserId) && roomUserSize > 2){
//				//随机获取房间内一个成员（不能是房主不能是自己）
//				List<String> userIds = room.getRoomUserIds();
//				userIds.remove(userName);
//				userIds.remove(ownerUserId);
//				
//				sdpAnswer = internalManager.subscribeBach(remoteName, userIds.get(0), sdpOffer, pid, rateLevel);
//			}else{
				sdpAnswer = internalManager.subscribe(remoteName, sdpOffer, pid, rateLevel);
//			}
			
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error subscribing to {}", userName, remoteName, e);
			notificationRoomHandler.onSubscribe(request, null, e);
		}
		if (sdpAnswer != null) {
			notificationRoomHandler.onSubscribe(request, sdpAnswer, null);
		}
	}

	/**
	 * @see RoomManager#unsubscribe(String, String)
	 *
	 * @param request
	 *            instance of {@link ParticipantRequest} POJO
	 */
	public void unsubscribe(String remoteName, ParticipantRequest request) {
		String pid = request.getParticipantId();
		String userName = null;
		boolean unsubscribed = false;
		try {
			userName = internalManager.getParticipantName(pid);
			internalManager.unsubscribe(remoteName, pid);
			unsubscribed = true;
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error unsubscribing from {}", userName, remoteName, e);
			notificationRoomHandler.onUnsubscribe(request, e);
		}
		if (unsubscribed) {
			notificationRoomHandler.onUnsubscribe(request, null);
		}
	}

	/**
	 * @see RoomManager#onIceCandidate(String, String, int, String, String)
	 */
	public void onIceCandidate(String endpointName, String candidate, int sdpMLineIndex, String sdpMid,
			ParticipantRequest request) {
		String pid = request.getParticipantId();
		String userName = null;
		try {
			userName = internalManager.getParticipantName(pid);
			internalManager.onIceCandidate(endpointName, candidate, sdpMLineIndex, sdpMid, request.getParticipantId());
			notificationRoomHandler.onRecvIceCandidate(request, null);
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error receiving ICE " + "candidate (epName={}, candidate={})", userName,
					endpointName, candidate, e);
			notificationRoomHandler.onRecvIceCandidate(request, e);
		}
	}

	/**
	 * Used by clients to send written messages to all other participants in the
	 * room.<br/>
	 * <strong>Side effects:</strong> The room event handler should acknowledge
	 * the client's request by sending an empty message. Should also send
	 * notifications to the all participants in the room with the message and
	 * its sender.
	 *
	 * @param message
	 *            message contents
	 * @param userName
	 *            name or identifier of the user in the room
	 * @param roomName
	 *            room's name
	 * @param request
	 *            instance of {@link ParticipantRequest} POJO
	 */
	public void sendMessage(String message, String userName, String roomName, ParticipantRequest request) {
		log.debug("Request [SEND_MESSAGE] message={} ({})", message, request);
		try {
			if (!internalManager.getParticipantName(request.getParticipantId()).equals(userName)) {
				throw new RoomException(Code.USER_NOT_FOUND_ERROR_CODE,
						"Provided username '" + userName + "' differs from the participant's name");
			}
			if (!internalManager.getRoomName(request.getParticipantId()).equals(roomName)) {
				throw new RoomException(Code.ROOM_NOT_FOUND_ERROR_CODE,
						"Provided room name '" + roomName + "' differs from the participant's room");
			}
			notificationRoomHandler.onSendMessage(request, message, userName, roomName,
					internalManager.getParticipants(roomName), null);
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error sending message", userName, e);
			notificationRoomHandler.onSendMessage(request, null, null, null, null, e);
		}
	}

	/**
	 * 新添加的事件处理
	 * start
	 */
	
	public void toggleSomeoneVideo(JsonObject params, String userName, String roomName, ParticipantRequest request) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		String participantId = null;
		Set<UserParticipant> existingParticipants = null;
		try {
			existingParticipants = internalManager.getParticipants(roomName);
			
			for (UserParticipant userParticipant : existingParticipants) {
				if (userParticipant.getUserName().equals(userName)) {
					participantId = userParticipant.getParticipantId();
				}
			}
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByPersonal(ProtocolElements.TOGGLE_SOMEONE_VIDEO_METHOD, notifParams, resultParam, 
					request, participantId, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByPersonal(ProtocolElements.TOGGLE_SOMEONE_VIDEO_METHOD, notifParams, resultParam, 
					request, participantId, null);
		}
	}
	
	public void toggleSomeoneAudio(JsonObject params, String userName, String roomName, ParticipantRequest request) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		String participantId = null;
		Set<UserParticipant> existingParticipants = null;
		try {
			existingParticipants = internalManager.getParticipants(roomName);
			
			for (UserParticipant userParticipant : existingParticipants) {
				if (userParticipant.getUserName().equals(userName)) {
					participantId = userParticipant.getParticipantId();
				}
			}
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByPersonal(ProtocolElements.TOGGLE_SOMEONE_AUDIO_METHOD, notifParams, resultParam, 
					request, participantId, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByPersonal(ProtocolElements.TOGGLE_SOMEONE_AUDIO_METHOD, notifParams, resultParam, 
					request, participantId, null);
		}
	}
	
	public void allowOperate(JsonObject params, String userName, String roomName, ParticipantRequest request) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		String participantId = null;
		Set<UserParticipant> existingParticipants = null;
		try {
			existingParticipants = internalManager.getParticipants(roomName);
			
			for (UserParticipant userParticipant : existingParticipants) {
				if (userParticipant.getUserName().equals(userName)) {
					participantId = userParticipant.getParticipantId();
				}
			}
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByPersonal(ProtocolElements.ALLOW_OPERATE_METHOD, notifParams, resultParam, 
					request, participantId, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByPersonal(ProtocolElements.ALLOW_OPERATE_METHOD, notifParams, resultParam, 
					request, participantId, null);
		}
	}
	
	public void sendAnswer(JsonObject params, String userName, String roomName, ParticipantRequest request) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		String participantId = null;
		Set<UserParticipant> existingParticipants = null;
		try {
			existingParticipants = internalManager.getParticipants(roomName);
			for (UserParticipant userParticipant : existingParticipants) {
				if (userParticipant.getUserName().equals(userName)) {
					participantId = userParticipant.getParticipantId();
				}
			}
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByPersonal(ProtocolElements.SEND_ANSWER_METHOD, notifParams, resultParam, 
					request, participantId, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByPersonal(ProtocolElements.SEND_ANSWER_METHOD, notifParams, resultParam, 
					request, participantId, null);
		}
	}
	
	public void dissolutionMettingGroup(String roomName, ParticipantRequest request){
		JsonObject notifParams = new JsonObject();
		JsonObject resultParam = new JsonObject();
		Set<UserParticipant> existingParticipants = null;
		
		try {
			existingParticipants = internalManager.getParticipants(roomName);
			
			Room room = internalManager.getRoomByRoomName(roomName);
			
			room.setRoomLayoutStatus("allvideo");
			room.setRoomMeetingInfo(new HashMap<String, Object>());
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.DISSOLUTION_METTING_GROUP_METHOD, notifParams, 
					resultParam, request, existingParticipants, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.DISSOLUTION_METTING_GROUP_METHOD, notifParams, 
					resultParam, request, existingParticipants, null);
		}
	}
	
	public void raiseHands(JsonObject params, String roomName, ParticipantRequest request) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		Set<UserParticipant> existingParticipants = null;
		
		try {
			existingParticipants = internalManager.getParticipants(roomName);
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.RAISE_HANDS_METHOD, notifParams, 
					resultParam, request, existingParticipants, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.RAISE_HANDS_METHOD, notifParams, 
					resultParam, request, existingParticipants, null);
		}
	}
	
	public void disbandRoom(JsonObject params, String roomName, ParticipantRequest request) {
		JsonObject notifParams = new JsonObject();
		JsonObject resultParam = new JsonObject();
		Set<UserParticipant> existingParticipants = null;
		
		try {
			existingParticipants = internalManager.getParticipants(roomName);
			
			internalManager.destoryRoom(roomName);
//			internalManager.getRoomByRoomName(roomName).close();
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.DISBAND_ROOM_METHOD, notifParams, 
					resultParam, request, existingParticipants, e);
		}
		
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.DISBAND_ROOM_METHOD, notifParams, 
					resultParam, request, existingParticipants, null);
		}
	}
	
	public void updateRoomStatus(ParticipantRequest request, JsonObject params, String roomName, String roomLayoutStatus) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		Set<UserParticipant> existingParticipants = null;
		
		try {
			existingParticipants = internalManager.getParticipants(roomName);
			
			//存储房间内布局状态
			Room room = internalManager.getRoomByRoomName(roomName);
			room.setRoomLayoutStatus(roomLayoutStatus);
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.UPDATE_ROOM_STATUS_METHOD, notifParams, 
					resultParam, request, existingParticipants, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.UPDATE_ROOM_STATUS_METHOD, notifParams, 
					resultParam, request, existingParticipants, null);
		}
	}
	
	public void sendQuestion(JsonObject params, String roomName, ParticipantRequest request) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		Set<UserParticipant> existingParticipants = null;
		
		try {
			existingParticipants = internalManager.getParticipants(roomName);
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.SEND_QUESTION_METHOD, notifParams, 
					resultParam, request, existingParticipants, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.SEND_QUESTION_METHOD, notifParams, 
					resultParam, request, existingParticipants, null);
		}
	}
	
	public void closeAnswerSheet(JsonObject params, String roomName, ParticipantRequest request) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		Set<UserParticipant> existingParticipants = null;
		
		try {
			existingParticipants = internalManager.getParticipants(roomName);
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.CLOSE_ANSWER_SHEET_METHOD, notifParams, 
					resultParam, request, existingParticipants, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.CLOSE_ANSWER_SHEET_METHOD, notifParams, 
					resultParam, request, existingParticipants, null);
		}
	}
	
	public void oneKeyOperation(JsonObject params, String roomName, ParticipantRequest request) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		Set<UserParticipant> existingParticipants = null;
		
		try {
			existingParticipants = internalManager.getParticipants(roomName);
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.ONE_KEY_OPERATION_METHOD, notifParams, 
					resultParam, request, existingParticipants, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.ONE_KEY_OPERATION_METHOD, notifParams, 
					resultParam, request, existingParticipants, null);
		}
	}
	
	public void sendAnswerQuestionResult(JsonObject params, String roomName, ParticipantRequest request) {
		JsonObject notifParams = params;
		JsonObject resultParam = new JsonObject();
		Set<UserParticipant> existingParticipants = null;
		
		try {
			existingParticipants = internalManager.getParticipants(roomName);
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.SEND_ANSWER_QUESTION_RESULT_METHOD, notifParams, 
					resultParam, request, existingParticipants, e);
		}
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByManyPeople("other", ProtocolElements.SEND_ANSWER_QUESTION_RESULT_METHOD, notifParams, 
					resultParam, request, existingParticipants, null);
		}
	}
	
	public void createMettingGroup(String userId, String roomName, String classMettingId, String classId, String cmName, String number, 
			String groupInfo, ParticipantRequest request){
		JsonObject notifParams = new JsonObject();
		JsonObject resultParam = new JsonObject();
		JSONObject mettingGroupData = null;
		Set<UserParticipant> existingParticipants = null;
		Map<String, String> paramsMap = new HashMap<>();
		
		try {
			
			log.info("createMettingGroup:{}","根据roomName获取房间成员");
			
			existingParticipants = internalManager.getParticipants(roomName);
			
			log.info("createMettingGroup:{}","根据roomName获取房间成员完毕");
			
			log.info("createMettingGroup:{}",existingParticipants);
			
			log.info("createMettingGroup:{}","根据roomName获取房间对象");
			
			Room room = internalManager.getRoomByRoomName(roomName);
			
			log.info("createMettingGroup:{}","根据roomName获取房间对象完毕");
			
			log.info("createMettingGroup:{}",room);
			
			paramsMap.put("userId", userId);
			paramsMap.put("cmName", cmName);
			paramsMap.put("number", number);
			paramsMap.put("groupInfo", groupInfo);
			
			log.info("createMettingGroup:{},{}", "classMettingId", classMettingId);
			
			log.info("方法:{}中, {}的值: {}", "createMettingGroup", "env.getProperty(\"classin.url\")" , env.getProperty("classin.url"));
			if(classMettingId.equals("")){
				paramsMap.put("classId", classId);
				
				log.info("createMettingGroup:{}", "开始addGroup");
				
//				mettingGroupData = HttpUtil.classPost(Constants.CLASS_IN_URL, "addGroup", paramsMap); // addGroup时classin服务返回的resultData
				mettingGroupData = HttpUtil.classPost(env.getProperty("classin.url"), "addGroup", paramsMap); // addGroup时classin服务返回的resultData
				log.info("createMettingGroup:{}", "addGroup完毕");
				
				log.info("createMettingGroup:{}", mettingGroupData);
				log.info("createMettingGroup:{}", mettingGroupData.toJSONString());
				
			}else{
				paramsMap.put("cmId", classMettingId);
//				mettingGroupData = HttpUtil.classPost(Constants.CLASS_IN_URL, "updateGroup", paramsMap);
				mettingGroupData = HttpUtil.classPost(env.getProperty("classin.url"), "updateGroup", paramsMap);
			}
			
			resultParam.addProperty("createStatus", false);
			if(mettingGroupData != null){
				
				log.info("createMettingGroup:{},{}", "mettingGroupData.getString(\"code\")", mettingGroupData.getString("code"));
				
				if(mettingGroupData.getString("code").equals("0")){
					resultParam.addProperty("createStatus", true);
					//添加房间信息 分组信息
					JSONObject meetingGroup = mettingGroupData.getJSONObject("data");
					
					log.info("createMettingGroup:{},{}", "meetingGroup", meetingGroup);
					
					Iterator<Object> jsonArray = JSONArray.parseArray(meetingGroup.get("groupDetail").toString()).iterator();
					List<Map<String,Object>> roomMettingGroup = new ArrayList<Map<String, Object>>();
					
					while (jsonArray.hasNext()) {
						JSONObject groupObject = (JSONObject) jsonArray.next();
						Map<String, Object> group = new HashMap<>();
						group.put("groupId", groupObject.get("groupId").toString());
						group.put("members", groupObject.get("members").toString());
						group.put("groupName", groupObject.get("groupName").toString());
						roomMettingGroup.add(group);
					}
					
					room.setRoomGroupDetails(roomMettingGroup);
					room.setRoomLayoutStatus("group");
					
					Map<String,Object> roomMeeting = new HashMap<String, Object>();
					roomMeeting.put("cmId", meetingGroup.getString("cmId"));
					roomMeeting.put("cmName", meetingGroup.getString("cmName"));
					room.setRoomMeetingInfo(roomMeeting);
				}else{
					resultParam.addProperty("code", mettingGroupData.getString("code"));
					notificationRoomHandler.onNoticeReturnResultByManyPeople("oather", ProtocolElements.CREATE_METTING_GROUP_METHOD, notifParams, resultParam, 
							request, existingParticipants, null);
				}
			}else{
				resultParam.addProperty("code", 11007);
				notificationRoomHandler.onNoticeReturnResultByManyPeople("oather", ProtocolElements.CREATE_METTING_GROUP_METHOD, notifParams, resultParam, 
						request, existingParticipants, null);
			}
			
		}catch (RoomException e) {
			notificationRoomHandler.onNoticeReturnResultByManyPeople("oather", ProtocolElements.CREATE_METTING_GROUP_METHOD, notifParams, resultParam, 
					request, existingParticipants, e);
		}
		
		if(existingParticipants != null){
			notificationRoomHandler.onNoticeReturnResultByManyPeople("oather", ProtocolElements.CREATE_METTING_GROUP_METHOD, notifParams, resultParam, 
					request, existingParticipants, null);
		}
	}
	
	public void delMettingGroup(String userId, String roomId, String classMettingId, ParticipantRequest request){
		JsonObject resultParam = new JsonObject();
		Boolean delStatus = false;
		String ownerUser = "";
		Map<String, String> paramsMap = new HashMap<>();
		
		paramsMap.put("cmId", classMettingId);
		paramsMap.put("roomId", roomId);
		
		try{
			Room room = internalManager.getRoomByRoomName(roomId);
			ownerUser = room.getOwner();
			if(userId.equals(ownerUser)){
				paramsMap.put("userId", userId);
//				JSONObject mettingGroupData = HttpUtil.classPost(Constants.CLASS_IN_URL, "delGroup", paramsMap);
				JSONObject mettingGroupData = HttpUtil.classPost(env.getProperty("classin.url"), "delGroup", paramsMap);
				
				if(mettingGroupData != null && mettingGroupData.getString("code").equals("0")){
					delStatus =  true;
				}
			}
		}catch (Exception e) {
			
		}
		
		resultParam.addProperty("delStatus", delStatus);
		
		notificationRoomHandler.onReturnResult(resultParam, request);
	}
	

	public void teacherGoingGroup(String status, String roomName, String currentGroupId, ParticipantRequest request){
		JsonObject resultParam = new JsonObject();
		JsonObject notifParams = new JsonObject();
		Set<UserParticipant> noticeParticipants = new HashSet<>();
		notifParams.addProperty("status", status);
		
		try{
			Room room = internalManager.getRoomByRoomName(roomName);
			notifParams.addProperty("ownerId", room.getOwner());
			Set<UserParticipant> remainingParticipants = internalManager.getParticipants(roomName);
			
			String noticeMembers = "";
			List<Map<String,Object>> roomGroupDetails = room.getRoomGroupDetails();
			for(Map<String, Object> group : roomGroupDetails){
				if(currentGroupId.equals(group.get("groupId"))){
					noticeMembers = group.get("members").toString();break;
				}
			}
			String[] noticeUserId = noticeMembers.split(",");
			for(UserParticipant participant : remainingParticipants){
				for(String uid : noticeUserId){
					if(participant.getUserName().equals(uid)){
						noticeParticipants.add(participant);
					}
				}
			}
			//设置老师进入的分组id
			if(status.equals("out")){
				currentGroupId = "";
			}
			room.setTeacherInGroupId(currentGroupId);
			resultParam.addProperty("teacherIntoStatus", true);
		}catch (Exception e) {
			resultParam.addProperty("teacherIntoStatus", false);
		}
		
		notificationRoomHandler.onNoticeReturnResultByManyPeople("all", ProtocolElements.TEACHER_GOING_GROUP_METHOD, notifParams, resultParam, request, noticeParticipants, null);
	}
	/**
	 * 获取房间状态
	 * @param roomId	房间id
	 * @return
	 */
	public String queryRoomLayoutStatus(String roomId){
		try{
			Room room = internalManager.getRoomByRoomName(roomId);
			return room.getRoomLayoutStatus();
		}catch (Exception e) {
			return "";
		}
	}
	
	/**
	 * 获取房间创建者userId
	 * @param roomId	房间id
	 * @return
	 */
	public String queryRoomOwnerUserId(String roomId){
		try{
			Room room = internalManager.getRoomByRoomName(roomId);
			return room.getOwner();
		}catch (Exception e) {
			return "";
		}
	}
	
	public void setPeopleVideoStatus(String userId, String videoStatus, ParticipantRequest request){
		JsonObject resultParam = new JsonObject();
		String pid = request.getParticipantId();
		try {
			internalManager.setParticipantVideoStatus(userId, videoStatus , pid);
			resultParam.addProperty("setStatus", true);
		} catch (RoomException e) {
			resultParam.addProperty("setStatus", false);
		}
		notificationRoomHandler.onReturnResult(resultParam, request);
	}
	
	public void rollCall(String rollCallName, String roomName, String roomLayoutStatus, ParticipantRequest request){
		String pid = request.getParticipantId();
		Set<UserParticipant> remainingParticipants = null;
	    
		String rollCallId = internalManager.getParticipantIdByName(rollCallName, pid);
		String role = internalManager.getParticipantRole(rollCallId);
	    
		if(role.equals("rollcalled")){
			log.warn("{} has long been called ", rollCallName );
			notificationRoomHandler.onParticipantRollCall(request, null, null, null);
		}else{
			try {
				internalManager.setParticipantRole(rollCallName, "rollcalled" , pid);
				remainingParticipants = internalManager.getParticipants(roomName);
				
				//存储房间内布局状态
				Room room = internalManager.getRoomByRoomName(roomName);
				room.setRoomLayoutStatus(roomLayoutStatus);
			} catch (RoomException e) {
				notificationRoomHandler.onParticipantRollCall(request, null, null, e);
			}
			if (remainingParticipants != null) {
				notificationRoomHandler.onParticipantRollCall(request, rollCallName, remainingParticipants, null);
			}
		}
	}

	public void sitDown(String sitDownName, String roomName, String roomLayoutStatus, ParticipantRequest request){
		String pid = request.getParticipantId();
		  
		Set<UserParticipant> remainingParticipants = null;

		String sitDownID = internalManager.getParticipantIdByName(sitDownName, pid);
		String role = internalManager.getParticipantRole(sitDownID);
		if(role.equals("normal")){
			log.warn("{} has long been called to sitdown ", sitDownName );
			notificationRoomHandler.onParticipantSitDown(request, null, null, null);
		}else{
			try {
				internalManager.setParticipantRole(sitDownName, "normal",pid);	
				remainingParticipants = internalManager.getParticipants(roomName);
				
				//存储房间内布局状态
				Room room = internalManager.getRoomByRoomName(roomName);
				room.setRoomLayoutStatus(roomLayoutStatus);
			} catch (RoomException e) {
				notificationRoomHandler.onParticipantSitDown(request, null, null, e);
			}
			if (remainingParticipants != null) {
				notificationRoomHandler.onParticipantSitDown(request, sitDownName, remainingParticipants, null);
			}
		}
	}
	
	/** 新添加的事件处理 end */
	
	
	
	// ----------------- APPLICATION-ORIGINATED REQUESTS ------------
	/**
	 * @see RoomManager#close()
	 */
	@PreDestroy
	public void close() {
		if (!internalManager.isClosed()) {
			internalManager.close();
		}
	}

	/**
	 * @see RoomManager#getRooms()
	 */
	public Set<String> getRooms() {
		return internalManager.getRooms();
	}

	/**
	 * @see RoomManager#getParticipants(String)
	 */
	public Set<UserParticipant> getParticipants(String roomName) throws RoomException {
		return internalManager.getParticipants(roomName);
	}

	/**
	 * @see RoomManager#getPublishers(String)
	 */
	public Set<UserParticipant> getPublishers(String roomName) throws RoomException {
		return internalManager.getPublishers(roomName);
	}

	/**
	 * @see RoomManager#getSubscribers(String)
	 */
	public Set<UserParticipant> getSubscribers(String roomName) throws RoomException {
		return internalManager.getSubscribers(roomName);
	}

	/**
	 * @see RoomManager#getPeerPublishers(String)
	 */
	public Set<UserParticipant> getPeerPublishers(String participantId) throws RoomException {
		return internalManager.getPeerPublishers(participantId);
	}

	/**
	 * @see RoomManager#getPeerSubscribers(String)
	 */
	public Set<UserParticipant> getPeerSubscribers(String participantId) throws RoomException {
		return internalManager.getPeerSubscribers(participantId);
	}
	
	/**
	 * 判断房间是否存在
	 */
	public boolean isRoomExist(String roomid){
		return internalManager.isRoomExist(roomid);
	}

	/**
	 * @see RoomManager#createRoom(KurentoClientSessionInfo)
	 */
	public void createRoom(KurentoClientSessionInfo kcSessionInfo) throws RoomException {
		internalManager.createRoom(kcSessionInfo);
	}
	
	public void createRoom(String uid, String roomId) throws RoomException {
		internalManager.createRoom(uid, roomId);
	}

	public Set<String> queryOnGoingRoom(){
		return internalManager.queryOnGoingRoom();
	}
	
	/**
	 * @see RoomManager#getPipeline(String)
	 */
	public MediaPipeline getPipeline(String participantId) throws RoomException {
		return internalManager.getPipeline(participantId);
	}

	/**
	 * Application-originated request to remove a participant from the room.
	 * <br/>
	 * <strong>Side effects:</strong> The room event handler should notify the
	 * user that she has been evicted. Should also send notifications to all
	 * other participants about the one that's just been evicted.
	 *
	 * @see RoomManager#leaveRoom(String)
	 */
	public void evictParticipant(String participantId) throws RoomException {
		UserParticipant participant = internalManager.getParticipantInfo(participantId);
		if(participant != null){
			Set<UserParticipant> remainingParticipants = internalManager.leaveRoom(participantId);
			notificationRoomHandler.onParticipantLeft(participant.getUserName(), remainingParticipants);
			notificationRoomHandler.onParticipantEvicted(participant);
		}
	}

	/**
	 * @see RoomManager#closeRoom(String)
	 */
	public void closeRoom(String roomName) throws RoomException {
		Set<UserParticipant> participants = internalManager.closeRoom(roomName);
		notificationRoomHandler.onRoomClosed(roomName, participants);
	}
	
	public void closeRoomWithoutNoti(String roomName) throws RoomException {
		internalManager.closeRoom(roomName);
	}

	/**
	 * @see RoomManager#generatePublishOffer(String)
	 */
	public String generatePublishOffer(String participantId) throws RoomException {
		return internalManager.generatePublishOffer(participantId);
	}

	/**
	 * @see RoomManager#addMediaElement(String, MediaElement)
	 */
	public void addMediaElement(String participantId, MediaElement element) throws RoomException {
		internalManager.addMediaElement(participantId, element);
	}

	/**
	 * @see RoomManager#addMediaElement(String, MediaElement, MediaType)
	 */
	public void addMediaElement(String participantId, MediaElement element, MediaType type) throws RoomException {
		internalManager.addMediaElement(participantId, element, type);
	}

	/**
	 * @see RoomManager#removeMediaElement(String, MediaElement)
	 */
	public void removeMediaElement(String participantId, MediaElement element) throws RoomException {
		internalManager.removeMediaElement(participantId, element);
	}

	/**
	 * @see RoomManager#mutePublishedMedia(MutedMediaType, String)
	 */
	public void mutePublishedMedia(MutedMediaType muteType, String participantId) throws RoomException {
		internalManager.mutePublishedMedia(muteType, participantId);
	}

	/**
	 * @see RoomManager#unmutePublishedMedia(String)
	 */
	public void unmutePublishedMedia(String participantId) throws RoomException {
		internalManager.unmutePublishedMedia(participantId);
	}

	/**
	 * @see RoomManager#muteSubscribedMedia(String, MutedMediaType, String)
	 */
	public void muteSubscribedMedia(String remoteName, MutedMediaType muteType, String participantId)
			throws RoomException {
		internalManager.muteSubscribedMedia(remoteName, muteType, participantId);
	}

	/**
	 * @see RoomManager#unmuteSubscribedMedia(String, String)
	 */
	public void unmuteSubscribedMedia(String remoteName, String participantId) throws RoomException {
		internalManager.unmuteSubscribedMedia(remoteName, participantId);
	}

	public RoomManager getRoomManager() {
		return internalManager;
	}

	public void sendError(ParticipantRequest request, String userName, RoomException roomException) {
		notificationRoomHandler.sendError(request, userName, roomException);
	}

	public void recordRoom(String userName, String roomName, ParticipantRequest participantRequest) {
		try{
		    internalManager.recordRoom(userName, roomName);
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error record for room {}", userName, roomName, e);
			notificationRoomHandler.onRecordRoom(participantRequest, e);
		}
		notificationRoomHandler.onRecordRoom(participantRequest, null);
	}

	public void stopRecordRoom(String userName, String roomName, ParticipantRequest participantRequest) {
		try {
			internalManager.stopRecordRoom(userName, roomName);
		} catch (RoomException e) {
			log.warn("PARTICIPANT {}: Error record for room {}", userName, roomName, e);
			notificationRoomHandler.onStopRecordRoom(participantRequest, e);
		}
		notificationRoomHandler.onStopRecordRoom(participantRequest, null);
	}

}
