package org.kurento.room.internal;

import java.util.List;
import java.util.Map;
import java.util.Set;
import org.kurento.client.IceCandidate;
import org.kurento.room.api.NotificationRoomHandler;
import org.kurento.room.api.UserNotificationService;
import org.kurento.room.api.pojo.ParticipantRequest;
import org.kurento.room.api.pojo.UserParticipant;
import org.kurento.room.exception.RoomException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.alibaba.fastjson.JSON;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

/**
 * Default implementation that assumes that JSON-RPC messages specification was
 * used for the client-server communications.
 *
 * @author <a href="mailto:rvlad@naevatec.com">Radu Tom Vlad</a>
 */
public class DefaultNotificationRoomHandler implements NotificationRoomHandler {
	private static final Logger log = LoggerFactory.getLogger(DefaultNotificationRoomHandler.class);
	private UserNotificationService notifService;

	public DefaultNotificationRoomHandler(UserNotificationService notifService) {
		this.notifService = notifService;
	}

	@Override
	public void onRoomClosed(String roomName, Set<UserParticipant> participants) {
		JsonObject notifParams = new JsonObject();
		notifParams.addProperty(ProtocolElements.ROOMCLOSED_ROOM_PARAM, roomName);
		for (UserParticipant participant : participants) {
			notifService.sendNotification(participant.getParticipantId(), ProtocolElements.ROOMCLOSED_METHOD, notifParams);
		}
	}

	@Override
	public void onParticipantJoined(ParticipantRequest request, String roomName, String ownerUser, String newUserName, String newUserRole,
			Set<UserParticipant> existingParticipants, RoomException error, Map<String,Object> roomMeetingInfo, String roomLayoutStatus,
			String teacherInGroupId, List<Map<String,Object>> roomGroupDetails) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}

		JsonArray result = new JsonArray();
		for (UserParticipant participant : existingParticipants) {
			JsonObject participantJson = new JsonObject();
			participantJson.addProperty(ProtocolElements.JOINROOM_PEERID_PARAM, participant.getUserName());
			if (participant.isStreaming()) {
				JsonObject stream = new JsonObject();
				stream.addProperty(ProtocolElements.JOINROOM_PEERSTREAMID_PARAM, "webcam");
				JsonArray streamsArray = new JsonArray();
				streamsArray.add(stream);
				participantJson.add(ProtocolElements.JOINROOM_PEERSTREAMS_PARAM, streamsArray);
			}
			participantJson.addProperty(ProtocolElements.PARTICIPANTJOINED_USER_ROLE, participant.getRole());
			result.add(participantJson);

			JsonObject notifParams = new JsonObject();
			notifParams.addProperty(ProtocolElements.PARTICIPANTJOINED_USER_PARAM, newUserName);
			notifParams.addProperty("teacherInGroupId", teacherInGroupId);
			//
			notifParams.addProperty(ProtocolElements.PARTICIPANTJOINED_USER_ROLE, newUserRole);
			notifService.sendNotification(participant.getParticipantId(), ProtocolElements.PARTICIPANTJOINED_METHOD,
					notifParams);
		}
		JsonObject resultData = new JsonObject();
		resultData.add("userlist", result);
		resultData.addProperty("roomLayoutStatus", roomLayoutStatus);
		String userInGroup = "";
		String userInGroupId = "";
		if(!roomLayoutStatus.equals("")){
			resultData.addProperty("roomMeetingInfo", JSON.toJSONString(roomMeetingInfo));
			resultData.addProperty(ProtocolElements.TEACHER_IN_GROUPID, teacherInGroupId);
			
			if(ownerUser.equals(newUserName)){
				resultData.addProperty("isOwner", true);
				
				if(teacherInGroupId.equals("")){
					userInGroup = "";
					resultData.addProperty("teacherAllGroup", JSON.toJSONString(roomGroupDetails));
				}else{
					for(Map<String, Object> group : roomGroupDetails){
						if(teacherInGroupId.equals(group.get("groupId").toString())){
							userInGroup = group.get("members").toString();
							userInGroup = userInGroup + "," + ownerUser;
							userInGroupId = group.get("groupId").toString();
							break;
						}
					}
				}
			}else{
				resultData.addProperty("isOwner", false);
				
				for(Map<String, Object> group : roomGroupDetails){
					if(group.get("members").toString().indexOf(newUserName) != -1){
						userInGroupId = group.get("groupId").toString();
						if(teacherInGroupId.equals(group.get("groupId"))){
							userInGroup = group.get("members").toString();
							userInGroup = userInGroup + "," + ownerUser;
							break;
						}else{
							userInGroup = group.get("members").toString();
							break;
						}
					}
				}
			}
		}
		
		resultData.addProperty("userInGroupId", userInGroupId);
		resultData.addProperty("userInGroupMember", userInGroup);
		
		notifService.sendResponse(request, resultData);
	}
	
	
	
	
	@Override
	public void onParticipantLeft(ParticipantRequest request, String userName,
			Set<UserParticipant> remainingParticipants, RoomException error) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}

		JsonObject params = new JsonObject();
		params.addProperty(ProtocolElements.PARTICIPANTLEFT_NAME_PARAM, userName);
		for (UserParticipant participant : remainingParticipants) {
			notifService.sendNotification(participant.getParticipantId(), ProtocolElements.PARTICIPANTLEFT_METHOD,
					params);
		}

		notifService.sendResponse(request, new JsonObject());
		notifService.closeSession(request);
	}

	@Override
	public void onPublishMedia(ParticipantRequest request, String publisherName, String role, String roomStatus, String teacherGroupId, String sdpAnswer,
			Set<UserParticipant> participants, String userId, RoomException error) {
	    if (error != null) {
	        notifService.sendErrorResponse(request, null, error);
	        return;
	      }
	      JsonObject result = new JsonObject();
	      result.addProperty(ProtocolElements.PUBLISHVIDEO_SDPANSWER_PARAM, sdpAnswer);
	      notifService.sendResponse(request, result);

	      JsonObject params = new JsonObject();
	      params.addProperty(ProtocolElements.PARTICIPANTPUBLISHED_USER_PARAM, publisherName);
	      params.addProperty(ProtocolElements.PARTICIPANTPUBLISHED_USER_ROLE, role);
	      params.addProperty(ProtocolElements.VIDEO_STATUS, roomStatus);
	      params.addProperty(ProtocolElements.TEACHER_IN_GROUPID, teacherGroupId);
	      JsonObject stream = new JsonObject();
	      stream.addProperty(ProtocolElements.PARTICIPANTPUBLISHED_STREAMID_PARAM, "webcam");
	      JsonArray streamsArray = new JsonArray();
	      streamsArray.add(stream);
	      params.add(ProtocolElements.PARTICIPANTPUBLISHED_STREAMS_PARAM, streamsArray);
	      
	      log.info("方法:{}:::::::::::::::::通知了{}个人","onPublishMedia", participants.size());
	      for (UserParticipant participant : participants) {
	        if(participant.getParticipantId().equals(request.getParticipantId())) {
	      	  //params.addProperty(ProtocolElements.PARTICIPANTPUBLISHED_USER_PARAM, publisherName + "_self");
	      	  continue;
	        }else{
	        	log.info("方法:{}::::::::::::::::,{}收到了,{}的{}", "onPublishMedia", participant.getUserName(), userId, role);
	    	    notifService.sendNotification(participant.getParticipantId(),
	    		        ProtocolElements.PARTICIPANTPUBLISHED_METHOD, params);
	        }
	      }
	}
	
	@Override
	public void onUnpublishMedia(ParticipantRequest request, String publisherName, Set<UserParticipant> participants,
			RoomException error) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}
		notifService.sendResponse(request, new JsonObject());

		JsonObject params = new JsonObject();
		params.addProperty(ProtocolElements.PARTICIPANTUNPUBLISHED_NAME_PARAM, publisherName);

		for (UserParticipant participant : participants) {
			if (participant.getParticipantId().equals(request.getParticipantId())) {
				continue;
			} else {
				notifService.sendNotification(participant.getParticipantId(),
						ProtocolElements.PARTICIPANTUNPUBLISHED_METHOD, params);
			}
		}
	}

	@Override
	public void onSubscribe(ParticipantRequest request, String sdpAnswer, RoomException error) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}
		JsonObject result = new JsonObject();
		result.addProperty(ProtocolElements.RECEIVEVIDEO_SDPANSWER_PARAM, sdpAnswer);
		notifService.sendResponse(request, result);
	}
	
	@Override
	public void onViewRoom(ParticipantRequest request, String sdpAnswer, RoomException error) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}
		JsonObject result = new JsonObject();
		result.addProperty(ProtocolElements.VIEWROOMVIDEO_SDPANSWER_PARAM, sdpAnswer);
		notifService.sendResponse(request, result);
	}

	@Override
	public void onUnsubscribe(ParticipantRequest request, RoomException error) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}
		notifService.sendResponse(request, new JsonObject());
	}

	@Override
	public void onSendMessage(ParticipantRequest request, String message, String userName, String roomName,
			Set<UserParticipant> participants, RoomException error) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}
		notifService.sendResponse(request, new JsonObject());

		JsonObject params = new JsonObject();
		params.addProperty(ProtocolElements.PARTICIPANTSENDMESSAGE_ROOM_PARAM, roomName);
		params.addProperty(ProtocolElements.PARTICIPANTSENDMESSAGE_USER_PARAM, userName);
		params.addProperty(ProtocolElements.PARTICIPANTSENDMESSAGE_MESSAGE_PARAM, message);

		for (UserParticipant participant : participants) {
			notifService.sendNotification(participant.getParticipantId(),
					ProtocolElements.PARTICIPANTSENDMESSAGE_METHOD, params);
		}
	}

	@Override
	public void onRecvIceCandidate(ParticipantRequest request, RoomException error) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}

		notifService.sendResponse(request, new JsonObject());
	}

	@Override
	public void onParticipantLeft(String userName, Set<UserParticipant> remainingParticipants) {
		JsonObject params = new JsonObject();
		params.addProperty(ProtocolElements.PARTICIPANTLEFT_NAME_PARAM, userName);
		for (UserParticipant participant : remainingParticipants) {
			notifService.sendNotification(participant.getParticipantId(), ProtocolElements.PARTICIPANTLEFT_METHOD,
					params);
		}
	}

	@Override
	public void onParticipantEvicted(UserParticipant participant) {
		notifService.sendNotification(participant.getParticipantId(), ProtocolElements.PARTICIPANTEVICTED_METHOD,
				new JsonObject());
	}

	// ------------ EVENTS FROM ROOM HANDLER -----

	@Override
	public void onIceCandidate(String roomName, String participantId, String endpointName, IceCandidate candidate) {
		JsonObject params = new JsonObject();
		params.addProperty(ProtocolElements.ICECANDIDATE_EPNAME_PARAM, endpointName);
		params.addProperty(ProtocolElements.ICECANDIDATE_SDPMLINEINDEX_PARAM, candidate.getSdpMLineIndex());
		params.addProperty(ProtocolElements.ICECANDIDATE_SDPMID_PARAM, candidate.getSdpMid());
		params.addProperty(ProtocolElements.ICECANDIDATE_CANDIDATE_PARAM, candidate.getCandidate());
		notifService.sendNotification(participantId, ProtocolElements.ICECANDIDATE_METHOD, params);
	}

	@Override
	public void onPipelineError(String roomName, Set<String> participantIds, String description) {
		JsonObject notifParams = new JsonObject();
		notifParams.addProperty(ProtocolElements.MEDIAERROR_ERROR_PARAM, description);
		for (String pid : participantIds) {
			notifService.sendNotification(pid, ProtocolElements.MEDIAERROR_METHOD, notifParams);
		}
	}

	@Override
	public void onMediaElementError(String roomName, String participantId, String description) {
		JsonObject notifParams = new JsonObject();
		notifParams.addProperty(ProtocolElements.MEDIAERROR_ERROR_PARAM, description);
		notifService.sendNotification(participantId, ProtocolElements.MEDIAERROR_METHOD, notifParams);
	}

	@Override
	public void sendError(ParticipantRequest request, String userName, RoomException roomException){
		notifService.sendErrorResponse(request, null, roomException);
	}

	@Override
	public void onRecordRoom(ParticipantRequest request, RoomException error) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}
		notifService.sendResponse(request, new JsonObject());
	}

	@Override
	public void onStopRecordRoom(ParticipantRequest request, RoomException error) {
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);
			return;
		}
		notifService.sendResponse(request, new JsonObject());
	}
	
	/**
	 * 直接返回不做通知的
	 * @param resultParam	返回要传递的参数
	 * @param request		返回request
	 * @return
	 */
	@Override
	public void onReturnResult(JsonObject resultParam, ParticipantRequest request){
		notifService.sendResponse(request, resultParam);
	}
	
	/**
	 * 通知单个人
	 * @param eventName		要通知的事件名称
	 * @param notifParams	通知要传递的参数
	 * @param resultParam	返回要传递的参数
	 * @param request		返回request
	 * @param participantId	单人的id
	 * @return
	 */
	@Override
	public void onNoticeReturnResultByPersonal(String eventName, JsonObject notifParams, JsonObject resultParam, ParticipantRequest request, 
			String participantId, RoomException error){
		if(error != null){
			notifService.sendErrorResponse(request, null, error);return;
		}
		notifService.sendNotification(participantId, eventName, notifParams);
		
		notifService.sendResponse(request, resultParam);
	}
	
	/**
	 * 通知多个人
	 * @param peopleNuber	人数状态(all-所有人,other-其他人)
	 * @param eventName		要通知的事件名称
	 * @param notifParams	通知要传递的参数
	 * @param resultParam	返回要传递的参数
	 * @param request		返回request
	 * @param existingParticipants	多人的id
	 * @return
	 */
	@Override
	public void onNoticeReturnResultByManyPeople(String peopleNuber, String eventName, JsonObject notifParams, JsonObject resultParam, 
			ParticipantRequest request, Set<UserParticipant> existingParticipants, RoomException error) {
		if(error != null){
			notifService.sendErrorResponse(request, null, error);return;
		}
		if(peopleNuber.equals("all")){
			if(existingParticipants.size() > 0){
				for (UserParticipant userParticipant : existingParticipants) {
					notifService.sendNotification(userParticipant.getParticipantId(), eventName, notifParams);
				}
			}
		}else{
			if(existingParticipants.size() > 0){
				for (UserParticipant userParticipant : existingParticipants) {
					if (request.getParticipantId().equals(userParticipant.getParticipantId())){
						continue;
					}
					notifService.sendNotification(userParticipant.getParticipantId(), eventName, notifParams);
				}
			}
		}
		notifService.sendResponse(request, resultParam);
	}
	
	@Override
	public void onParticipantRollCall(ParticipantRequest request, String userName,
			Set<UserParticipant> participants, RoomException error){
		if (error != null) {
			notifService.sendErrorResponse(request, null, error);return;
		}
	    
		if(participants == null){
			JsonObject param = new JsonObject();
			param.addProperty(ProtocolElements.ROLL_CALL_RESULT, "alreadyCalled");
	        notifService.sendResponse(request, param);
	        return;
	    }
	    
	    JsonObject params = new JsonObject();
	    params.addProperty(ProtocolElements.PARTICIPANTLEFT_NAME_PARAM, userName);
	    for (UserParticipant participant : participants) {
	    	if(participant.getUserName().equals(userName)){
	    		  log.info("方法:{}:::::::::::::::::{}RollCall了", "onParticipantRollCall", userName);
	    	      notifService.sendNotification(participant.getParticipantId(),
	    	              ProtocolElements.ROLL_CALL_METHOD, params);
	    	      break;
	    	}
	    }
	    
	    notifService.sendResponse(request, new JsonObject());
	  }
	  
	@Override
	public void onParticipantSitDown(ParticipantRequest request, String userName,
			Set<UserParticipant> participants, RoomException error){
		if(error != null){
			notifService.sendErrorResponse(request, null, error);return;
		}
		  
		if(participants == null){
			JsonObject param = new JsonObject();
			param.addProperty(ProtocolElements.SIT_DOWN_RESULT, "alreadysitdown");
			notifService.sendResponse(request, param);
			return;
		}
		  
		JsonObject params = new JsonObject();
		params.addProperty(ProtocolElements.PARTICIPANTLEFT_NAME_PARAM, userName);
		for (UserParticipant participant : participants) {
			if(participant.getUserName().equals(userName)){
				log.info("方法:{}:::::::::::::::::{}SitDown了", "onParticipantSitDown", userName);
				notifService.sendNotification(participant.getParticipantId(), ProtocolElements.SIT_DOWN_METHOD, params);break;
			}
		}

		notifService.sendResponse(request, new JsonObject());
	}
}
